<?php

namespace Bixo\Account\Http\Controllers;

use App\Http\Controllers\APIController as BaseController;
use Bixo\Account\Http\Requests\EscrowRequest;
use Bixo\Account\Interfaces\EscrowRepositoryInterface;
use Bixo\Account\Models\Escrow;
use Bixo\Account\Forms\Escrow as Form;

/**
 * APIController  class for escrow.
 */
class EscrowAPIController extends BaseController
{

    /**
     * Initialize escrow resource controller.
     *
     * @param type EscrowRepositoryInterface $escrow
     *
     * @return null
     */
    public function __construct(EscrowRepositoryInterface $escrow)
    {
        parent::__construct();
        $this->repository = $escrow;
        $this->repository
            ->pushCriteria(\Litepie\Repository\Criteria\RequestCriteria::class)
            ->pushCriteria(\Bixo\Account\Repositories\Criteria\EscrowResourceCriteria::class);
    }

    /**
     * Display a list of escrow.
     *
     * @return Response
     */
    public function index(EscrowRequest $request)
    {
        return $this->repository
            ->setPresenter(\Bixo\Account\Repositories\Presenter\EscrowPresenter::class)
            ->paginate();
    }

    /**
     * Display escrow.
     *
     * @param Request $request
     * @param Model   $escrow
     *
     * @return Response
     */
    public function show(EscrowRequest $request, Escrow $escrow)
    {
        return $escrow->setPresenter(\Bixo\Account\Repositories\Presenter\EscrowListPresenter::class);
        ;
    }

    /**
     * Create new escrow.
     *
     * @param Request $request
     *
     * @return Response
     */
    public function store(EscrowRequest $request)
    {
        try {
            $data              = $request->all();
            $data['user_id']   = user_id();
            $data['user_type'] = user_type();
            $data              = $this->repository->create($data);
            $message           = trans('messages.success.created', ['Module' => trans('account::escrow.name')]);
            $code              = 204;
            $status            = 'success';
            $url               = guard_url('account/escrow/' . $escrow->getRouteKey());
        } catch (Exception $e) {
            $message = $e->getMessage();
            $code    = 400;
            $status  = 'error';
            $url     = guard_url('account/escrow');
        }
        return compact('data', 'message', 'code', 'status', 'url');
    }

    /**
     * Update the escrow.
     *
     * @param Request $request
     * @param Model   $escrow
     *
     * @return Response
     */
    public function update(EscrowRequest $request, Escrow $escrow)
    {
        try {
            $data = $request->all();

            $escrow->update($data);
            $message = trans('messages.success.updated', ['Module' => trans('account::escrow.name')]);
            $code    = 204;
            $status  = 'success';
            $url     = guard_url('account/escrow/' . $escrow->getRouteKey());
        } catch (Exception $e) {
            $message = $e->getMessage();
            $code    = 400;
            $status  = 'error';
            $url     = guard_url('account/escrow/' . $escrow->getRouteKey());
        }
        return compact('data', 'message', 'code', 'status', 'url');
    }

    /**
     * Remove the escrow.
     *
     * @param Model   $escrow
     *
     * @return Response
     */
    public function destroy(EscrowRequest $request, Escrow $escrow)
    {
        try {
            $escrow->delete();
            $message = trans('messages.success.deleted', ['Module' => trans('account::escrow.name')]);
            $code    = 202;
            $status  = 'success';
            $url     = guard_url('account/escrow/0');
        } catch (Exception $e) {
            $message = $e->getMessage();
            $code    = 400;
            $status  = 'error';
            $url     = guard_url('account/escrow/' . $escrow->getRouteKey());
        }
        return compact('message', 'code', 'status', 'url');
    }

    /**
     * Return the form elements as json.
     *
     * @param String   $element
     *
     * @return json
     */
    public function form($element = 'fields')
    {
        $form = new Form();
        return $form->form($element, true);
    }

}
