<?php

namespace Bixo\Account\Http\Controllers;

use App\Http\Controllers\ResourceController as BaseController;
use Form;
use Bixo\Account\Http\Requests\EscrowDetailsRequest;
use Bixo\Account\Interfaces\EscrowDetailsRepositoryInterface;
use Bixo\Account\Models\EscrowDetails;

/**
 * Resource controller class for escrow_details.
 */
class EscrowDetailsResourceController extends BaseController
{

    /**
     * Initialize escrow_details resource controller.
     *
     * @param type EscrowDetailsRepositoryInterface $escrow_details
     *
     * @return null
     */
    public function __construct(EscrowDetailsRepositoryInterface $escrow_details)
    {
        parent::__construct();
        $this->repository = $escrow_details;
        $this->repository
            ->pushCriteria(\Litepie\Repository\Criteria\RequestCriteria::class)
            ->pushCriteria(\Bixo\Account\Repositories\Criteria\EscrowDetailsResourceCriteria::class);
    }

    /**
     * Display a list of escrow_details.
     *
     * @return Response
     */
    public function index(EscrowDetailsRequest $request)
    {
        $view = $this->response->theme->listView();

        if ($this->response->typeIs('json')) {
            $function = camel_case('get-' . $view);
            return $this->repository
                ->setPresenter(\Bixo\Account\Repositories\Presenter\EscrowDetailsPresenter::class)
                ->$function();
        }

        $escrow_details = $this->repository->paginate();

        return $this->response->setMetaTitle(trans('account::escrow_details.names'))
            ->view('account::escrow_details.index', true)
            ->data(compact('escrow_details', 'view'))
            ->output();
    }

    /**
     * Display escrow_details.
     *
     * @param Request $request
     * @param Model   $escrow_details
     *
     * @return Response
     */
    public function show(EscrowDetailsRequest $request, EscrowDetails $escrow_details)
    {

        if ($escrow_details->exists) {
            $view = 'account::escrow_details.show';
        } else {
            $view = 'account::escrow_details.new';
        }

        return $this->response->setMetaTitle(trans('app.view') . ' ' . trans('account::escrow_details.name'))
            ->data(compact('escrow_details'))
            ->view($view, true)
            ->output();
    }

    /**
     * Show the form for creating a new escrow_details.
     *
     * @param Request $request
     *
     * @return Response
     */
    public function create(EscrowDetailsRequest $request)
    {

        $escrow_details = $this->repository->newInstance([]);
        return $this->response->setMetaTitle(trans('app.new') . ' ' . trans('account::escrow_details.name')) 
            ->view('account::escrow_details.create', true) 
            ->data(compact('escrow_details'))
            ->output();
    }

    /**
     * Create new escrow_details.
     *
     * @param Request $request
     *
     * @return Response
     */
    public function store(EscrowDetailsRequest $request)
    {
        try {
            $attributes              = $request->all();
            $attributes['user_id']   = user_id();
            $attributes['user_type'] = user_type();
            $escrow_details                 = $this->repository->create($attributes);

            return $this->response->message(trans('messages.success.created', ['Module' => trans('account::escrow_details.name')]))
                ->code(204)
                ->status('success')
                ->url(guard_url('account/escrow_details/' . $escrow_details->getRouteKey()))
                ->redirect();
        } catch (Exception $e) {
            return $this->response->message($e->getMessage())
                ->code(400)
                ->status('error')
                ->url(guard_url('/account/escrow_details'))
                ->redirect();
        }

    }

    /**
     * Show escrow_details for editing.
     *
     * @param Request $request
     * @param Model   $escrow_details
     *
     * @return Response
     */
    public function edit(EscrowDetailsRequest $request, EscrowDetails $escrow_details)
    {
        return $this->response->setMetaTitle(trans('app.edit') . ' ' . trans('account::escrow_details.name'))
            ->view('account::escrow_details.edit', true)
            ->data(compact('escrow_details'))
            ->output();
    }

    /**
     * Update the escrow_details.
     *
     * @param Request $request
     * @param Model   $escrow_details
     *
     * @return Response
     */
    public function update(EscrowDetailsRequest $request, EscrowDetails $escrow_details)
    {
        try {
            $attributes = $request->all();

            $escrow_details->update($attributes);
            return $this->response->message(trans('messages.success.updated', ['Module' => trans('account::escrow_details.name')]))
                ->code(204)
                ->status('success')
                ->url(guard_url('account/escrow_details/' . $escrow_details->getRouteKey()))
                ->redirect();
        } catch (Exception $e) {
            return $this->response->message($e->getMessage())
                ->code(400)
                ->status('error')
                ->url(guard_url('account/escrow_details/' . $escrow_details->getRouteKey()))
                ->redirect();
        }

    }

    /**
     * Remove the escrow_details.
     *
     * @param Model   $escrow_details
     *
     * @return Response
     */
    public function destroy(EscrowDetailsRequest $request, EscrowDetails $escrow_details)
    {
        try {

            $escrow_details->delete();
            return $this->response->message(trans('messages.success.deleted', ['Module' => trans('account::escrow_details.name')]))
                ->code(202)
                ->status('success')
                ->url(guard_url('account/escrow_details/0'))
                ->redirect();

        } catch (Exception $e) {

            return $this->response->message($e->getMessage())
                ->code(400)
                ->status('error')
                ->url(guard_url('account/escrow_details/' . $escrow_details->getRouteKey()))
                ->redirect();
        }

    }

    /**
     * Remove multiple escrow_details.
     *
     * @param Model   $escrow_details
     *
     * @return Response
     */
    public function delete(EscrowDetailsRequest $request, $type)
    {
        try {
            $ids = hashids_decode($request->input('ids'));

            if ($type == 'purge') {
                $this->repository->purge($ids);
            } else {
                $this->repository->delete($ids);
            }

            return $this->response->message(trans('messages.success.deleted', ['Module' => trans('account::escrow_details.name')]))
                ->status("success")
                ->code(202)
                ->url(guard_url('account/escrow_details'))
                ->redirect();

        } catch (Exception $e) {

            return $this->response->message($e->getMessage())
                ->status("error")
                ->code(400)
                ->url(guard_url('/account/escrow_details'))
                ->redirect();
        }

    }

    /**
     * Restore deleted escrow_details.
     *
     * @param Model   $escrow_details
     *
     * @return Response
     */
    public function restore(EscrowDetailsRequest $request)
    {
        try {
            $ids = hashids_decode($request->input('ids'));
            $this->repository->restore($ids);

            return $this->response->message(trans('messages.success.restore', ['Module' => trans('account::escrow_details.name')]))
                ->status("success")
                ->code(202)
                ->url(guard_url('/account/escrow_details'))
                ->redirect();

        } catch (Exception $e) {

            return $this->response->message($e->getMessage())
                ->status("error")
                ->code(400)
                ->url(guard_url('/account/escrow_details/'))
                ->redirect();
        }

    }

}
