<?php

namespace Bixo\Account\Http\Controllers;

use App\Http\Controllers\ResourceController as BaseController;
use Form;
use Bixo\Account\Http\Requests\EscrowRequest;
use Bixo\Account\Interfaces\EscrowRepositoryInterface;
use Bixo\Account\Models\Escrow;

/**
 * Resource controller class for escrow.
 */
class EscrowResourceController extends BaseController
{

    /**
     * Initialize escrow resource controller.
     *
     * @param type EscrowRepositoryInterface $escrow
     *
     * @return null
     */
    public function __construct(EscrowRepositoryInterface $escrow)
    {
        parent::__construct();
        $this->repository = $escrow;
        $this->repository
            ->pushCriteria(\Litepie\Repository\Criteria\RequestCriteria::class)
            ->pushCriteria(\Bixo\Account\Repositories\Criteria\EscrowResourceCriteria::class);
    }

    /**
     * Display a list of escrow.
     *
     * @return Response
     */
    public function index(EscrowRequest $request)
    {
        $view = $this->response->theme->listView();

        if ($this->response->typeIs('json')) {
            $function = camel_case('get-' . $view);
            return $this->repository
                ->setPresenter(\Bixo\Account\Repositories\Presenter\EscrowPresenter::class)
                ->$function();
        }

        $escrows = $this->repository->paginate();

        return $this->response->setMetaTitle(trans('account::escrow.names'))
            ->view('account::escrow.index', true)
            ->data(compact('escrows', 'view'))
            ->output();
    }

    /**
     * Display escrow.
     *
     * @param Request $request
     * @param Model   $escrow
     *
     * @return Response
     */
    public function show(EscrowRequest $request, Escrow $escrow)
    {

        if ($escrow->exists) {
            $view = 'account::escrow.show';
        } else {
            $view = 'account::escrow.new';
        }

        return $this->response->setMetaTitle(trans('app.view') . ' ' . trans('account::escrow.name'))
            ->data(compact('escrow'))
            ->view($view, true)
            ->output();
    }

    /**
     * Show the form for creating a new escrow.
     *
     * @param Request $request
     *
     * @return Response
     */
    public function create(EscrowRequest $request)
    {

        $escrow = $this->repository->newInstance([]);
        return $this->response->setMetaTitle(trans('app.new') . ' ' . trans('account::escrow.name')) 
            ->view('account::escrow.create', true) 
            ->data(compact('escrow'))
            ->output();
    }

    /**
     * Create new escrow.
     *
     * @param Request $request
     *
     * @return Response
     */
    public function store(EscrowRequest $request)
    {
        try {
            $attributes              = $request->all();
            $attributes['user_id']   = user_id();
            $attributes['user_type'] = user_type();
            $escrow                 = $this->repository->create($attributes);

            return $this->response->message(trans('messages.success.created', ['Module' => trans('account::escrow.name')]))
                ->code(204)
                ->status('success')
                ->url(guard_url('account/escrow/' . $escrow->getRouteKey()))
                ->redirect();
        } catch (Exception $e) {
            return $this->response->message($e->getMessage())
                ->code(400)
                ->status('error')
                ->url(guard_url('/account/escrow'))
                ->redirect();
        }

    }

    /**
     * Show escrow for editing.
     *
     * @param Request $request
     * @param Model   $escrow
     *
     * @return Response
     */
    public function edit(EscrowRequest $request, Escrow $escrow)
    {
        return $this->response->setMetaTitle(trans('app.edit') . ' ' . trans('account::escrow.name'))
            ->view('account::escrow.edit', true)
            ->data(compact('escrow'))
            ->output();
    }

    /**
     * Update the escrow.
     *
     * @param Request $request
     * @param Model   $escrow
     *
     * @return Response
     */
    public function update(EscrowRequest $request, Escrow $escrow)
    {
        try {
            $attributes = $request->all();

            $escrow->update($attributes);
            return $this->response->message(trans('messages.success.updated', ['Module' => trans('account::escrow.name')]))
                ->code(204)
                ->status('success')
                ->url(guard_url('account/escrow/' . $escrow->getRouteKey()))
                ->redirect();
        } catch (Exception $e) {
            return $this->response->message($e->getMessage())
                ->code(400)
                ->status('error')
                ->url(guard_url('account/escrow/' . $escrow->getRouteKey()))
                ->redirect();
        }

    }

    /**
     * Remove the escrow.
     *
     * @param Model   $escrow
     *
     * @return Response
     */
    public function destroy(EscrowRequest $request, Escrow $escrow)
    {
        try {

            $escrow->delete();
            return $this->response->message(trans('messages.success.deleted', ['Module' => trans('account::escrow.name')]))
                ->code(202)
                ->status('success')
                ->url(guard_url('account/escrow/0'))
                ->redirect();

        } catch (Exception $e) {

            return $this->response->message($e->getMessage())
                ->code(400)
                ->status('error')
                ->url(guard_url('account/escrow/' . $escrow->getRouteKey()))
                ->redirect();
        }

    }

    /**
     * Remove multiple escrow.
     *
     * @param Model   $escrow
     *
     * @return Response
     */
    public function delete(EscrowRequest $request, $type)
    {
        try {
            $ids = hashids_decode($request->input('ids'));

            if ($type == 'purge') {
                $this->repository->purge($ids);
            } else {
                $this->repository->delete($ids);
            }

            return $this->response->message(trans('messages.success.deleted', ['Module' => trans('account::escrow.name')]))
                ->status("success")
                ->code(202)
                ->url(guard_url('account/escrow'))
                ->redirect();

        } catch (Exception $e) {

            return $this->response->message($e->getMessage())
                ->status("error")
                ->code(400)
                ->url(guard_url('/account/escrow'))
                ->redirect();
        }

    }

    /**
     * Restore deleted escrows.
     *
     * @param Model   $escrow
     *
     * @return Response
     */
    public function restore(EscrowRequest $request)
    {
        try {
            $ids = hashids_decode($request->input('ids'));
            $this->repository->restore($ids);

            return $this->response->message(trans('messages.success.restore', ['Module' => trans('account::escrow.name')]))
                ->status("success")
                ->code(202)
                ->url(guard_url('/account/escrow'))
                ->redirect();

        } catch (Exception $e) {

            return $this->response->message($e->getMessage())
                ->status("error")
                ->code(400)
                ->url(guard_url('/account/escrow/'))
                ->redirect();
        }

    }

}
