<?php

namespace Annotators\Annotator\Http\Controllers;

use Exception;
use Litepie\Http\Controllers\ResourceController as BaseController;
use Litepie\Database\RequestScope;
use Annotators\Annotator\Forms\Annotator as AnnotatorForm;
use Annotators\Annotator\Http\Requests\AnnotatorRequest;
use Annotators\Annotator\Http\Resources\AnnotatorResource;
use Annotators\Annotator\Http\Resources\AnnotatorsCollection;
use Annotators\Annotator\Models\Annotator;
use Annotators\Annotator\Scopes\AnnotatorResourceScope;


/**
 * Resource controller class for annotator.
 */
class AnnotatorResourceController extends BaseController
{

    /**
     * Initialize annotator resource controller.
     *
     *
     * @return null
     */
    public function __construct()
    {
        parent::__construct();
        $this->form = AnnotatorForm::grouped(false)
                        ->setAttributes()
                        ->toArray();
        $this->modules = $this->modules(config('annotators.annotator.modules'), 'annotator', guard_url('annotator'));
    }

    /**
     * Display a list of annotator.
     *
     * @return Response
     */
    public function index(AnnotatorRequest $request)
    {

        $pageLimit = $request->input('pageLimit', config('database.pagination.limit'));
        $page = Annotator::pushScope(new RequestScope())
            ->pushScope(new AnnotatorResourceScope())
            ->simplePaginate($pageLimit);

        $data = new AnnotatorsCollection($page);

        $form = $this->form;
        $modules = $this->modules;

        return $this->response->setMetaTitle(trans('annotator::annotator.names'))
            ->view('annotator::annotator.index')
            ->data(compact('data', 'modules', 'form'))
            ->output();

    }

    /**
     * Display annotator.
     *
     * @param Request $request
     * @param Model   $annotator
     *
     * @return Response
     */
    public function show(AnnotatorRequest $request, Annotator $model)
    {
        $form = $this->form;
        $modules = $this->modules;
        $data = new AnnotatorResource($model);

        return $this->response
            ->setMetaTitle(trans('app.view') . ' ' . trans('annotator::annotator.name'))
            ->data(compact('data', 'form', 'modules'))
            ->view('annotator::annotator.show')
            ->output();

    }

    /**
     * Show the form for creating a new annotator.
     *
     * @param Request $request
     *
     * @return Response
     */
    public function create(AnnotatorRequest $request, Annotator $model)
    {
        $form = $this->form;
        $modules = $this->modules;
        $data = new AnnotatorResource($model);
        return $this->response->setMetaTitle(trans('app.new') . ' ' . trans('annotator::annotator.name'))
            ->view('annotator::annotator.create')
            ->data(compact('data', 'form', 'modules'))
            ->output();

    }

    /**
     * Create new annotator.
     *
     * @param Request $request
     *
     * @return Response
     */
    public function store(AnnotatorRequest $request, Annotator $model)
    {
        try {
            $attributes = $request->all();
            $attributes['user_id'] = user_id();
            $attributes['user_type'] = user_type();
            $model = $model->create($attributes);
            $data = new AnnotatorResource($model);

            return $this->response->message(trans('messages.success.created', ['Module' => trans('annotator::annotator.name')]))
                ->code(204)
                ->data(compact('data'))
                ->status('success')
                ->url(guard_url('annotator/annotator/' . $model->getRouteKey()))
                ->redirect();
        } catch (Exception $e) {
            return $this->response->message($e->getMessage())
                ->code(400)
                ->status('error')
                ->url(guard_url('/annotator/annotator'))
                ->redirect();
        }

    }

    /**
     * Show annotator for editing.
     *
     * @param Request $request
     * @param Model   $annotator
     *
     * @return Response
     */
    public function edit(AnnotatorRequest $request, Annotator $model)
    {
        $form = $this->form;
        $modules = $this->modules;
        $data = new AnnotatorResource($model);

        return $this->response->setMetaTitle(trans('app.edit') . ' ' . trans('annotator::annotator.name'))
            ->view('annotator::annotator.edit')
            ->data(compact('data', 'form', 'modules'))
            ->output();

    }

    /**
     * Update the annotator.
     *
     * @param Request $request
     * @param Model   $annotator
     *
     * @return Response
     */
    public function update(AnnotatorRequest $request, Annotator $model)
    {
        try {
            $attributes = $request->all();
            $model = $model->update($attributes);
            $data = new AnnotatorResource($model);

            return $this->response->message(trans('messages.success.updated', ['Module' => trans('annotator::annotator.name')]))
                ->code(204)
                ->status('success')
                ->data(compact('data'))
                ->url(guard_url('annotator/annotator/' . $model->getRouteKey()))
                ->redirect();
        } catch (Exception $e) {
            return $this->response->message($e->getMessage())
                ->code(400)
                ->status('error')
                ->url(guard_url('annotator/annotator/' .  $model->getRouteKey()))
                ->redirect();
        }

    }

    /**
     * Remove the annotator.
     *
     * @param Model   $annotator
     *
     * @return Response
     */
    public function destroy(AnnotatorRequest $request, Annotator $model)
    {
        try {
            $model->delete();
            $data = new AnnotatorResource($model);

            return $this->response->message(trans('messages.success.deleted', ['Module' => trans('annotator::annotator.name')]))
                ->code(202)
                ->status('success')
                ->data(compact('data'))
                ->url(guard_url('annotator/annotator/0'))
                ->redirect();

        } catch (Exception $e) {

            return $this->response->message($e->getMessage())
                ->code(400)
                ->status('error')
                ->url(guard_url('annotator/annotator/' .  $model->getRouteKey()))
                ->redirect();
        }

    }
}
