<?php

namespace Zing\Bandwidth\Http\Controllers;

use Exception;
use Litepie\Http\Controllers\ResourceController as BaseController;
use Litepie\Repository\Filter\RequestFilter;
use Zing\Bandwidth\Forms\Bandwidth as BandwidthForm;
use Zing\Bandwidth\Http\Requests\BandwidthRequest;
use Zing\Bandwidth\Interfaces\BandwidthRepositoryInterface;
use Zing\Bandwidth\Repositories\Eloquent\Filters\BandwidthResourceFilter;
use Zing\Bandwidth\Repositories\Eloquent\Presenters\BandwidthListPresenter;

/**
 * Resource controller class for bandwidth.
 */
class BandwidthResourceController extends BaseController
{

    /**
     * Initialize bandwidth resource controller.
     *
     *
     * @return null
     */
    public function __construct(BandwidthRepositoryInterface $bandwidth)
    {
        parent::__construct();
        $this->form = BandwidthForm::setAttributes()->toArray();
        $this->modules = $this->modules(config('zing.bandwidth.modules'), 'bandwidth', guard_url('bandwidth'));
        $this->repository = $bandwidth;
    }

    /**
     * Display a list of bandwidth.
     *
     * @return Response
     */
    public function index(BandwidthRequest $request)
    {

        $pageLimit = $request->input('pageLimit', config('database.pagination.limit'));
        $data = $this->repository
            ->pushFilter(RequestFilter::class)
            ->pushFilter(BandwidthResourceFilter::class)
            ->setPresenter(BandwidthListPresenter::class)
            ->simplePaginate($pageLimit)
            // ->withQueryString()
            ->toArray();

        extract($data);
        $form = $this->form;
        $modules = $this->modules;

        return $this->response->setMetaTitle(trans('bandwidth::bandwidth.names'))
            ->view('bandwidth::bandwidth.index')
            ->data(compact('data', 'meta', 'links', 'modules', 'form'))
            ->output();
    }

    /**
     * Display bandwidth.
     *
     * @param Request $request
     * @param Model   $bandwidth
     *
     * @return Response
     */
    public function show(BandwidthRequest $request, BandwidthRepositoryInterface $repository)
    {
        $form = $this->form;
        $modules = $this->modules;
        $data = $repository->toArray();
        return $this->response
            ->setMetaTitle(trans('app.view') . ' ' . trans('bandwidth::bandwidth.name'))
            ->data(compact('data', 'form', 'modules', 'form'))
            ->view('bandwidth::bandwidth.show')
            ->output();
    }

    /**
     * Show the form for creating a new bandwidth.
     *
     * @param Request $request
     *x
     * @return Response
     */
    public function create(BandwidthRequest $request, BandwidthRepositoryInterface $repository)
    {
        $form = $this->form;
        $modules = $this->modules;
        $data = $repository->toArray();
        return $this->response->setMetaTitle(trans('app.new') . ' ' . trans('bandwidth::bandwidth.name'))
            ->view('bandwidth::bandwidth.create')
            ->data(compact('data', 'form', 'modules'))
            ->output();
    }

    /**
     * Create new bandwidth.
     *
     * @param Request $request
     *
     * @return Response
     */
    public function store(BandwidthRequest $request, BandwidthRepositoryInterface $repository)
    {
        try {
            $attributes = $request->all();
            $attributes['user_id'] = user_id();
            $attributes['user_type'] = user_type();
            $repository->create($attributes);
            $data = $repository->toArray();

            return $this->response->message(trans('messages.success.created', ['Module' => trans('bandwidth::bandwidth.name')]))
                ->code(204)
                ->data(compact('data'))
                ->status('success')
                ->url(guard_url('bandwidth/bandwidth/' . $data['id']))
                ->redirect();
        } catch (Exception $e) {
            return $this->response->message($e->getMessage())
                ->code(400)
                ->status('error')
                ->url(guard_url('/bandwidth/bandwidth'))
                ->redirect();
        }

    }

    /**
     * Show bandwidth for editing.
     *
     * @param Request $request
     * @param Model   $bandwidth
     *
     * @return Response
     */
    public function edit(BandwidthRequest $request, BandwidthRepositoryInterface $repository)
    {
        $form = $this->form;
        $modules = $this->modules;
        $data = $repository->toArray();

        return $this->response->setMetaTitle(trans('app.edit') . ' ' . trans('bandwidth::bandwidth.name'))
            ->view('bandwidth::bandwidth.edit')
            ->data(compact('data', 'form', 'modules'))
            ->output();
    }

    /**
     * Update the bandwidth.
     *
     * @param Request $request
     * @param Model   $bandwidth
     *
     * @return Response
     */
    public function update(BandwidthRequest $request, BandwidthRepositoryInterface $repository)
    {
        try {
            $attributes = $request->all();
            $repository->update($attributes);
            $data = $repository->toArray();

            return $this->response->message(trans('messages.success.updated', ['Module' => trans('bandwidth::bandwidth.name')]))
                ->code(204)
                ->status('success')
                ->data(compact('data'))
                ->url(guard_url('bandwidth/bandwidth/' . $data['id']))
                ->redirect();
        } catch (Exception $e) {
            return $this->response->message($e->getMessage())
                ->code(400)
                ->status('error')
                ->url(guard_url('bandwidth/bandwidth/' .  $repository->getRouteKey()))
                ->redirect();
        }

    }

    /**
     * Remove the bandwidth.
     *
     * @param Model   $bandwidth
     *
     * @return Response
     */
    public function destroy(BandwidthRequest $request, BandwidthRepositoryInterface $repository)
    {
        try {
            $repository->delete();
            $data = $repository->toArray();

            return $this->response->message(trans('messages.success.deleted', ['Module' => trans('bandwidth::bandwidth.name')]))
                ->code(202)
                ->status('success')
                ->data(compact('data'))
                ->url(guard_url('bandwidth/bandwidth/0'))
                ->redirect();

        } catch (Exception $e) {

            return $this->response->message($e->getMessage())
                ->code(400)
                ->status('error')
                ->url(guard_url('bandwidth/bandwidth/' .  $repository->getRouteKey()))
                ->redirect();
        }

    }
}
