<?php

namespace Liteerp\Branch\Http\Controllers;

use App\Http\Controllers\ResourceController as BaseController;
use Form;
use Liteerp\Branch\Http\Requests\BranchRequest;
use Liteerp\Branch\Interfaces\BranchRepositoryInterface;
use Liteerp\Branch\Models\Branch;

/**
 * Resource controller class for branch.
 */
class BranchResourceController extends BaseController
{

    /**
     * Initialize branch resource controller.
     *
     * @param type BranchRepositoryInterface $branch
     *
     * @return null
     */
    public function __construct(BranchRepositoryInterface $branch)
    {
        parent::__construct();
        $this->repository = $branch;
        $this->repository
            ->pushCriteria(\Litepie\Repository\Criteria\RequestCriteria::class)
            ->pushCriteria(\Liteerp\Branch\Repositories\Criteria\BranchResourceCriteria::class);
    }

    /**
     * Display a list of branch.
     *
     * @return Response
     */
    public function index(BranchRequest $request)
    {
        $view = $this->response->theme->listView();

        if ($this->response->typeIs('json')) {
            $function = camel_case('get-' . $view);
            return $this->repository
                ->setPresenter(\Liteerp\Branch\Repositories\Presenter\BranchPresenter::class)
                ->$function();
        }

        $branches = $this->repository->paginate();

        return $this->response->title(trans('branch::branch.names'))
            ->view('branch::branch.index', true)
            ->data(compact('branches'))
            ->output();
    }

    /**
     * Display branch.
     *
     * @param Request $request
     * @param Model   $branch
     *
     * @return Response
     */
    public function show(BranchRequest $request, Branch $branch)
    {

        if ($branch->exists) {
            $view = 'branch::branch.show';
        } else {
            $view = 'branch::branch.new';
        }

        return $this->response->title(trans('app.view') . ' ' . trans('branch::branch.name'))
            ->data(compact('branch'))
            ->view($view, true)
            ->output();
    }

    /**
     * Show the form for creating a new branch.
     *
     * @param Request $request
     *
     * @return Response
     */
    public function create(BranchRequest $request)
    {

        $branch = $this->repository->newInstance([]);
        return $this->response->title(trans('app.new') . ' ' . trans('branch::branch.name')) 
            ->view('branch::branch.create', true) 
            ->data(compact('branch'))
            ->output();
    }

    /**
     * Create new branch.
     *
     * @param Request $request
     *
     * @return Response
     */
    public function store(BranchRequest $request)
    {
        try {
            $attributes              = $request->all();
            $attributes['user_id']   = user_id();
            $attributes['user_type'] = user_type();
            $branch                 = $this->repository->create($attributes);

            return $this->response->message(trans('messages.success.created', ['Module' => trans('branch::branch.name')]))
                ->code(204)
                ->status('success')
                ->url(guard_url('branch/branch/' . $branch->getRouteKey()))
                ->redirect();
        } catch (Exception $e) {
            return $this->response->message($e->getMessage())
                ->code(400)
                ->status('error')
                ->url(guard_url('/branch/branch'))
                ->redirect();
        }

    }

    /**
     * Show branch for editing.
     *
     * @param Request $request
     * @param Model   $branch
     *
     * @return Response
     */
    public function edit(BranchRequest $request, Branch $branch)
    {
        return $this->response->title(trans('app.edit') . ' ' . trans('branch::branch.name'))
            ->view('branch::branch.edit', true)
            ->data(compact('branch'))
            ->output();
    }

    /**
     * Update the branch.
     *
     * @param Request $request
     * @param Model   $branch
     *
     * @return Response
     */
    public function update(BranchRequest $request, Branch $branch)
    {
        try {
            $attributes = $request->all();

            $branch->update($attributes);
            return $this->response->message(trans('messages.success.updated', ['Module' => trans('branch::branch.name')]))
                ->code(204)
                ->status('success')
                ->url(guard_url('branch/branch/' . $branch->getRouteKey()))
                ->redirect();
        } catch (Exception $e) {
            return $this->response->message($e->getMessage())
                ->code(400)
                ->status('error')
                ->url(guard_url('branch/branch/' . $branch->getRouteKey()))
                ->redirect();
        }

    }

    /**
     * Remove the branch.
     *
     * @param Model   $branch
     *
     * @return Response
     */
    public function destroy(BranchRequest $request, Branch $branch)
    {
        try {

            $branch->delete();
            return $this->response->message(trans('messages.success.deleted', ['Module' => trans('branch::branch.name')]))
                ->code(202)
                ->status('success')
                ->url(guard_url('branch/branch/0'))
                ->redirect();

        } catch (Exception $e) {

            return $this->response->message($e->getMessage())
                ->code(400)
                ->status('error')
                ->url(guard_url('branch/branch/' . $branch->getRouteKey()))
                ->redirect();
        }

    }

    /**
     * Remove multiple branch.
     *
     * @param Model   $branch
     *
     * @return Response
     */
    public function delete(BranchRequest $request, $type)
    {
        try {
            $ids = hashids_decode($request->input('ids'));

            if ($type == 'purge') {
                $this->repository->purge($ids);
            } else {
                $this->repository->delete($ids);
            }

            return $this->response->message(trans('messages.success.deleted', ['Module' => trans('branch::branch.name')]))
                ->status("success")
                ->code(202)
                ->url(guard_url('branch/branch'))
                ->redirect();

        } catch (Exception $e) {

            return $this->response->message($e->getMessage())
                ->status("error")
                ->code(400)
                ->url(guard_url('/branch/branch'))
                ->redirect();
        }

    }

    /**
     * Restore deleted branches.
     *
     * @param Model   $branch
     *
     * @return Response
     */
    public function restore(BranchRequest $request)
    {
        try {
            $ids = hashids_decode($request->input('ids'));
            $this->repository->restore($ids);

            return $this->response->message(trans('messages.success.restore', ['Module' => trans('branch::branch.name')]))
                ->status("success")
                ->code(202)
                ->url(guard_url('/branch/branch'))
                ->redirect();

        } catch (Exception $e) {

            return $this->response->message($e->getMessage())
                ->status("error")
                ->code(400)
                ->url(guard_url('/branch/branch/'))
                ->redirect();
        }

    }

}
