<?php

namespace Category\Category\Http\Controllers;

use App\Http\Controllers\AdminController as AdminController;
use Form;
use Category\Category\Http\Requests\CategoryAdminRequest;
use Category\Category\Interfaces\CategoryRepositoryInterface;
use Category\Category\Models\Category;

/**
 * Admin controller class.
 */
class CategoryAdminController extends AdminController
{
    /**
     * Initialize category controller.
     *
     * @param type CategoryRepositoryInterface $category
     *
     * @return type
     */
    public function __construct(CategoryRepositoryInterface $category)
    {
        parent::__construct();
        $this->repository = $category;
    }

    /**
     * Display a list of category.
     *
     * @return Response
     */
    public function index(CategoryAdminRequest $request)
    {
        $categories  = $this->repository->setPresenter('\\Category\\Category\\Repositories\\Presenter\\CategoryListPresenter')
                                            ->scopeQuery(function($query){
                                                return $query->orderBy('id','DESC');
                                            })->paginate();
        $this   ->theme->prependTitle(trans('category::category.names').' :: ');
        $view   = $this->theme->of('category::admin.category.index')->render();

        $this->responseCode = 200;
        $this->responseMessage = trans('messages.success.loaded', ['Module' => trans('category::category.name')]);
        $this->responseData = $categories['data'];
        $this->responseMeta = $categories['meta'];
        $this->responseView = $view;
        $this->responseRedirect = '';
        return $this->respond($request);
    }

    /**
     * Display category.
     *
     * @param Request $request
     * @param int     $id
     *
     * @return Response
     */
    public function show(CategoryAdminRequest $request, Category $category)
    {
        if (!$category->exists) {
            $this->responseCode = 404;
            $this->responseMessage = trans('messages.success.notfound', ['Module' => trans('category::category.name')]);
            $this->responseView = view('category::admin.category.new');
            return $this -> respond($request);
        }

        Form::populate($category);
        $this->responseCode = 200;
        $this->responseMessage = trans('messages.success.loaded', ['Module' => trans('category::category.name')]);
        $this->responseData = $category;
        $this->responseView = view('category::admin.category.show', compact('category'));
        return $this -> respond($request);
    }

    /**
     * Show the form for creating a new category.
     *
     * @param Request $request
     *
     * @return Response
     */
    public function create(CategoryAdminRequest $request)
    {

        $category = $this->repository->newInstance([]);

        Form::populate($category);

        $this->responseCode = 200;
        $this->responseMessage = trans('messages.success.loaded', ['Module' => trans('category::category.name')]);
        $this->responseData = $category;
        $this->responseView = view('category::admin.category.create', compact('category'));
        return $this -> respond($request);

    }

    /**
     * Create new category.
     *
     * @param Request $request
     *
     * @return Response
     */
    public function store(CategoryAdminRequest $request)
    {
        try {
            $attributes = $request->all();
            $attributes['user_id'] = user_id();
            $category = $this->repository->create($attributes);

            $this->responseCode = 201;
            $this->responseMessage = trans('messages.success.created', ['Module' => trans('category::category.name')]);
            $this->responseData = $category;
            $this->responseMeta = '';
            $this->responseRedirect = trans_url('/admin/category/category/'.$category->getRouteKey());
            $this->responseView = view('category::admin.category.create', compact('category'));

            return $this -> respond($request);

        } catch (Exception $e) {
            $this->responseCode = 400;
            $this->responseMessage = $e->getMessage();
            return $this -> respond($request);
        }
    }

    /**
     * Show category for editing.
     *
     * @param Request $request
     * @param int     $id
     *
     * @return Response
     */
    public function edit(CategoryAdminRequest $request, Category $category)
    {
        Form::populate($category);
        $this->responseCode = 200;
        $this->responseMessage = trans('messages.success.loaded', ['Module' => trans('category::category.name')]);
        $this->responseData = $category;
        $this->responseView = view('category::admin.category.edit', compact('category'));

        return $this -> respond($request);
    }

    /**
     * Update the category.
     *
     * @param Request $request
     * @param int     $id
     *
     * @return Response
     */
    public function update(CategoryAdminRequest $request, Category $category)
    {
        try {

            $attributes = $request->all();

            $category->update($attributes);

            $this->responseCode = 204;
            $this->responseMessage = trans('messages.success.updated', ['Module' => trans('category::category.name')]);
            $this->responseData = $category;
            $this->responseRedirect = trans_url('/admin/category/category/'.$category->getRouteKey());

            return $this -> respond($request);

        } catch (Exception $e) {

            $this->responseCode = 400;
            $this->responseMessage = $e->getMessage();
            $this->responseRedirect = trans_url('/admin/category/category/'.$category->getRouteKey());

            return $this -> respond($request);
        }
    }

    /**
     * Remove the category.
     *
     * @param int $id
     *
     * @return Response
     */
    public function destroy(CategoryAdminRequest $request, Category $category)
    {

        try {

            $t = $category->delete();

            $this->responseCode = 202;
            $this->responseMessage = trans('messages.success.deleted', ['Module' => trans('category::category.name')]);
            $this->responseRedirect = trans_url('/admin/category/category/0');

            return $this -> respond($request);

        } catch (Exception $e) {

            $this->responseCode = 400;
            $this->responseMessage = $e->getMessage();
            $this->responseRedirect = trans_url('/admin/category/category/'.$category->getRouteKey());

            return $this -> respond($request);

        }
    }
}
