<?php

namespace B2buy\Contract\Http\Controllers;

use Exception;
use Litepie\Http\Controllers\ResourceController as BaseController;
use Litepie\Repository\Filter\RequestFilter;
use B2buy\Contract\Forms\Bond as BondForm;
use B2buy\Contract\Http\Requests\BondRequest;
use B2buy\Contract\Interfaces\BondRepositoryInterface;
use B2buy\Contract\Repositories\Eloquent\Filters\BondResourceFilter;
use B2buy\Contract\Repositories\Eloquent\Presenters\BondListPresenter;

/**
 * Resource controller class for bond.
 */
class BondResourceController extends BaseController
{

    /**
     * Initialize bond resource controller.
     *
     *
     * @return null
     */
    public function __construct(BondRepositoryInterface $bond)
    {
        parent::__construct();
        $this->form = BondForm::setAttributes()->toArray();
        $this->modules = $this->modules(config('b2buy.contract.modules'), 'contract', guard_url('contract'));
        $this->repository = $bond;
    }

    /**
     * Display a list of bond.
     *
     * @return Response
     */
    public function index(BondRequest $request)
    {

        $pageLimit = $request->input('pageLimit', config('database.pagination.limit'));
        $data = $this->repository
            ->pushFilter(RequestFilter::class)
            ->pushFilter(BondResourceFilter::class)
            ->setPresenter(BondListPresenter::class)
            ->simplePaginate($pageLimit)
            // ->withQueryString()
            ->toArray();

        extract($data);
        $form = $this->form;
        $modules = $this->modules;

        return $this->response->setMetaTitle(trans('contract::bond.names'))
            ->view('contract::bond.index')
            ->data(compact('data', 'meta', 'links', 'modules', 'form'))
            ->output();
    }

    /**
     * Display bond.
     *
     * @param Request $request
     * @param Model   $bond
     *
     * @return Response
     */
    public function show(BondRequest $request, BondRepositoryInterface $repository)
    {
        $form = $this->form;
        $modules = $this->modules;
        $data = $repository->toArray();
        return $this->response
            ->setMetaTitle(trans('app.view') . ' ' . trans('contract::bond.name'))
            ->data(compact('data', 'form', 'modules', 'form'))
            ->view('contract::bond.show')
            ->output();
    }

    /**
     * Show the form for creating a new bond.
     *
     * @param Request $request
     *x
     * @return Response
     */
    public function create(BondRequest $request, BondRepositoryInterface $repository)
    {
        $form = $this->form;
        $modules = $this->modules;
        $data = $repository->toArray();
        return $this->response->setMetaTitle(trans('app.new') . ' ' . trans('contract::bond.name'))
            ->view('contract::bond.create')
            ->data(compact('data', 'form', 'modules'))
            ->output();
    }

    /**
     * Create new bond.
     *
     * @param Request $request
     *
     * @return Response
     */
    public function store(BondRequest $request, BondRepositoryInterface $repository)
    {
        try {
            $attributes = $request->all();
            $attributes['user_id'] = user_id();
            $attributes['user_type'] = user_type();
            $repository->create($attributes);
            $data = $repository->toArray();

            return $this->response->message(trans('messages.success.created', ['Module' => trans('contract::bond.name')]))
                ->code(204)
                ->data(compact('data'))
                ->status('success')
                ->url(guard_url('contract/bond/' . $data['id']))
                ->redirect();
        } catch (Exception $e) {
            return $this->response->message($e->getMessage())
                ->code(400)
                ->status('error')
                ->url(guard_url('/contract/bond'))
                ->redirect();
        }

    }

    /**
     * Show bond for editing.
     *
     * @param Request $request
     * @param Model   $bond
     *
     * @return Response
     */
    public function edit(BondRequest $request, BondRepositoryInterface $repository)
    {
        $form = $this->form;
        $modules = $this->modules;
        $data = $repository->toArray();

        return $this->response->setMetaTitle(trans('app.edit') . ' ' . trans('contract::bond.name'))
            ->view('contract::bond.edit')
            ->data(compact('data', 'form', 'modules'))
            ->output();
    }

    /**
     * Update the bond.
     *
     * @param Request $request
     * @param Model   $bond
     *
     * @return Response
     */
    public function update(BondRequest $request, BondRepositoryInterface $repository)
    {
        try {
            $attributes = $request->all();
            $repository->update($attributes);
            $data = $repository->toArray();

            return $this->response->message(trans('messages.success.updated', ['Module' => trans('contract::bond.name')]))
                ->code(204)
                ->status('success')
                ->data(compact('data'))
                ->url(guard_url('contract/bond/' . $data['id']))
                ->redirect();
        } catch (Exception $e) {
            return $this->response->message($e->getMessage())
                ->code(400)
                ->status('error')
                ->url(guard_url('contract/bond/' .  $repository->getRouteKey()))
                ->redirect();
        }

    }

    /**
     * Remove the bond.
     *
     * @param Model   $bond
     *
     * @return Response
     */
    public function destroy(BondRequest $request, BondRepositoryInterface $repository)
    {
        try {
            $repository->delete();
            $data = $repository->toArray();

            return $this->response->message(trans('messages.success.deleted', ['Module' => trans('contract::bond.name')]))
                ->code(202)
                ->status('success')
                ->data(compact('data'))
                ->url(guard_url('contract/bond/0'))
                ->redirect();

        } catch (Exception $e) {

            return $this->response->message($e->getMessage())
                ->code(400)
                ->status('error')
                ->url(guard_url('contract/bond/' .  $repository->getRouteKey()))
                ->redirect();
        }

    }
}
