<?php

namespace B2buy\Contract\Http\Controllers;

use Exception;
use Litepie\Http\Controllers\ResourceController as BaseController;
use Litepie\Repository\Filter\RequestFilter;
use B2buy\Contract\Forms\Contract as ContractForm;
use B2buy\Contract\Http\Requests\ContractRequest;
use B2buy\Contract\Interfaces\ContractRepositoryInterface;
use B2buy\Contract\Repositories\Eloquent\Filters\ContractResourceFilter;
use B2buy\Contract\Repositories\Eloquent\Presenters\ContractListPresenter;

/**
 * Resource controller class for contract.
 */
class ContractResourceController extends BaseController
{

    /**
     * Initialize contract resource controller.
     *
     *
     * @return null
     */
    public function __construct(ContractRepositoryInterface $contract)
    {
        parent::__construct();
        $this->form = ContractForm::setAttributes()->toArray();
        $this->modules = $this->modules(config('b2buy.contract.modules'), 'contract', guard_url('contract'));
        $this->repository = $contract;
    }

    /**
     * Display a list of contract.
     *
     * @return Response
     */
    public function index(ContractRequest $request)
    {

        $pageLimit = $request->input('pageLimit', config('database.pagination.limit'));
        $data = $this->repository
            ->pushFilter(RequestFilter::class)
            ->pushFilter(ContractResourceFilter::class)
            ->setPresenter(ContractListPresenter::class)
            ->simplePaginate($pageLimit)
            // ->withQueryString()
            ->toArray();

        extract($data);
        $form = $this->form;
        $modules = $this->modules;

        return $this->response->setMetaTitle(trans('contract::contract.names'))
            ->view('contract::contract.index')
            ->data(compact('data', 'meta', 'links', 'modules', 'form'))
            ->output();
    }

    /**
     * Display contract.
     *
     * @param Request $request
     * @param Model   $contract
     *
     * @return Response
     */
    public function show(ContractRequest $request, ContractRepositoryInterface $repository)
    {
        $form = $this->form;
        $modules = $this->modules;
        $data = $repository->toArray();
        return $this->response
            ->setMetaTitle(trans('app.view') . ' ' . trans('contract::contract.name'))
            ->data(compact('data', 'form', 'modules', 'form'))
            ->view('contract::contract.show')
            ->output();
    }

    /**
     * Show the form for creating a new contract.
     *
     * @param Request $request
     *x
     * @return Response
     */
    public function create(ContractRequest $request, ContractRepositoryInterface $repository)
    {
        $form = $this->form;
        $modules = $this->modules;
        $data = $repository->toArray();
        return $this->response->setMetaTitle(trans('app.new') . ' ' . trans('contract::contract.name'))
            ->view('contract::contract.create')
            ->data(compact('data', 'form', 'modules'))
            ->output();
    }

    /**
     * Create new contract.
     *
     * @param Request $request
     *
     * @return Response
     */
    public function store(ContractRequest $request, ContractRepositoryInterface $repository)
    {
        try {
            $attributes = $request->all();
            $attributes['user_id'] = user_id();
            $attributes['user_type'] = user_type();
            $repository->create($attributes);
            $data = $repository->toArray();

            return $this->response->message(trans('messages.success.created', ['Module' => trans('contract::contract.name')]))
                ->code(204)
                ->data(compact('data'))
                ->status('success')
                ->url(guard_url('contract/contract/' . $data['id']))
                ->redirect();
        } catch (Exception $e) {
            return $this->response->message($e->getMessage())
                ->code(400)
                ->status('error')
                ->url(guard_url('/contract/contract'))
                ->redirect();
        }

    }

    /**
     * Show contract for editing.
     *
     * @param Request $request
     * @param Model   $contract
     *
     * @return Response
     */
    public function edit(ContractRequest $request, ContractRepositoryInterface $repository)
    {
        $form = $this->form;
        $modules = $this->modules;
        $data = $repository->toArray();

        return $this->response->setMetaTitle(trans('app.edit') . ' ' . trans('contract::contract.name'))
            ->view('contract::contract.edit')
            ->data(compact('data', 'form', 'modules'))
            ->output();
    }

    /**
     * Update the contract.
     *
     * @param Request $request
     * @param Model   $contract
     *
     * @return Response
     */
    public function update(ContractRequest $request, ContractRepositoryInterface $repository)
    {
        try {
            $attributes = $request->all();
            $repository->update($attributes);
            $data = $repository->toArray();

            return $this->response->message(trans('messages.success.updated', ['Module' => trans('contract::contract.name')]))
                ->code(204)
                ->status('success')
                ->data(compact('data'))
                ->url(guard_url('contract/contract/' . $data['id']))
                ->redirect();
        } catch (Exception $e) {
            return $this->response->message($e->getMessage())
                ->code(400)
                ->status('error')
                ->url(guard_url('contract/contract/' .  $repository->getRouteKey()))
                ->redirect();
        }

    }

    /**
     * Remove the contract.
     *
     * @param Model   $contract
     *
     * @return Response
     */
    public function destroy(ContractRequest $request, ContractRepositoryInterface $repository)
    {
        try {
            $repository->delete();
            $data = $repository->toArray();

            return $this->response->message(trans('messages.success.deleted', ['Module' => trans('contract::contract.name')]))
                ->code(202)
                ->status('success')
                ->data(compact('data'))
                ->url(guard_url('contract/contract/0'))
                ->redirect();

        } catch (Exception $e) {

            return $this->response->message($e->getMessage())
                ->code(400)
                ->status('error')
                ->url(guard_url('contract/contract/' .  $repository->getRouteKey()))
                ->redirect();
        }

    }
}
