<?php

namespace Bixo\Database\Http\Controllers;

use App\Http\Controllers\APIController as BaseController;
use Bixo\Database\Http\Requests\CallRequest;
use Bixo\Database\Interfaces\CallRepositoryInterface;
use Bixo\Database\Models\Call;
use Bixo\Database\Forms\Call as Form;

/**
 * APIController  class for call.
 */
class CallAPIController extends BaseController
{

    /**
     * Initialize call resource controller.
     *
     * @param type CallRepositoryInterface $call
     *
     * @return null
     */
    public function __construct(CallRepositoryInterface $call)
    {
        parent::__construct();
        $this->repository = $call;
        $this->repository
            ->pushCriteria(\Litepie\Repository\Criteria\RequestCriteria::class)
            ->pushCriteria(\Bixo\Database\Repositories\Criteria\CallResourceCriteria::class);
    }

    /**
     * Display a list of call.
     *
     * @return Response
     */
    public function index(CallRequest $request)
    {
        return $this->repository
            ->setPresenter(\Bixo\Database\Repositories\Presenter\CallPresenter::class)
            ->paginate();
    }

    /**
     * Display call.
     *
     * @param Request $request
     * @param Model   $call
     *
     * @return Response
     */
    public function show(CallRequest $request, Call $call)
    {
        return $call->setPresenter(\Bixo\Database\Repositories\Presenter\CallListPresenter::class);
        ;
    }

    /**
     * Create new call.
     *
     * @param Request $request
     *
     * @return Response
     */
    public function store(CallRequest $request)
    {
        try {
            $data              = $request->all();
            $data['user_id']   = user_id();
            $data['user_type'] = user_type();
            $data              = $this->repository->create($data);
            $message           = trans('messages.success.created', ['Module' => trans('database::call.name')]);
            $code              = 204;
            $status            = 'success';
            $url               = guard_url('database/call/' . $call->getRouteKey());
        } catch (Exception $e) {
            $message = $e->getMessage();
            $code    = 400;
            $status  = 'error';
            $url     = guard_url('database/call');
        }
        return compact('data', 'message', 'code', 'status', 'url');
    }

    /**
     * Update the call.
     *
     * @param Request $request
     * @param Model   $call
     *
     * @return Response
     */
    public function update(CallRequest $request, Call $call)
    {
        try {
            $data = $request->all();

            $call->update($data);
            $message = trans('messages.success.updated', ['Module' => trans('database::call.name')]);
            $code    = 204;
            $status  = 'success';
            $url     = guard_url('database/call/' . $call->getRouteKey());
        } catch (Exception $e) {
            $message = $e->getMessage();
            $code    = 400;
            $status  = 'error';
            $url     = guard_url('database/call/' . $call->getRouteKey());
        }
        return compact('data', 'message', 'code', 'status', 'url');
    }

    /**
     * Remove the call.
     *
     * @param Model   $call
     *
     * @return Response
     */
    public function destroy(CallRequest $request, Call $call)
    {
        try {
            $call->delete();
            $message = trans('messages.success.deleted', ['Module' => trans('database::call.name')]);
            $code    = 202;
            $status  = 'success';
            $url     = guard_url('database/call/0');
        } catch (Exception $e) {
            $message = $e->getMessage();
            $code    = 400;
            $status  = 'error';
            $url     = guard_url('database/call/' . $call->getRouteKey());
        }
        return compact('message', 'code', 'status', 'url');
    }

    /**
     * Return the form elements as json.
     *
     * @param String   $element
     *
     * @return json
     */
    public function form($element = 'fields')
    {
        $form = new Form();
        return $form->form($element, true);
    }

}
