<?php

namespace Bixo\Database\Http\Controllers;

use App\Http\Controllers\ResourceController as BaseController;
use Form;
use Bixo\Database\Http\Requests\CallRequest;
use Bixo\Database\Interfaces\CallRepositoryInterface;
use Bixo\Database\Models\Call;

/**
 * Resource controller class for call.
 */
class CallResourceController extends BaseController
{

    /**
     * Initialize call resource controller.
     *
     * @param type CallRepositoryInterface $call
     *
     * @return null
     */
    public function __construct(CallRepositoryInterface $call)
    {
        parent::__construct();
        $this->repository = $call;
        $this->repository
            ->pushCriteria(\Litepie\Repository\Criteria\RequestCriteria::class)
            ->pushCriteria(\Bixo\Database\Repositories\Criteria\CallResourceCriteria::class);
    }

    /**
     * Display a list of call.
     *
     * @return Response
     */
    public function index(CallRequest $request)
    {
        $view = $this->response->theme->listView();

        if ($this->response->typeIs('json')) {
            $function = camel_case('get-' . $view);
            return $this->repository
                ->setPresenter(\Bixo\Database\Repositories\Presenter\CallPresenter::class)
                ->$function();
        }

        $calls = $this->repository->paginate();

        return $this->response->setMetaTitle(trans('database::call.names'))
            ->view('database::call.index', true)
            ->data(compact('calls', 'view'))
            ->output();
    }

    /**
     * Display call.
     *
     * @param Request $request
     * @param Model   $call
     *
     * @return Response
     */
    public function show(CallRequest $request, Call $call)
    {

        if ($call->exists) {
            $view = 'database::call.show';
        } else {
            $view = 'database::call.new';
        }

        return $this->response->setMetaTitle(trans('app.view') . ' ' . trans('database::call.name'))
            ->data(compact('call'))
            ->view($view, true)
            ->output();
    }

    /**
     * Show the form for creating a new call.
     *
     * @param Request $request
     *
     * @return Response
     */
    public function create(CallRequest $request)
    {

        $call = $this->repository->newInstance([]);
        return $this->response->setMetaTitle(trans('app.new') . ' ' . trans('database::call.name')) 
            ->view('database::call.create', true) 
            ->data(compact('call'))
            ->output();
    }

    /**
     * Create new call.
     *
     * @param Request $request
     *
     * @return Response
     */
    public function store(CallRequest $request)
    {
        try {
            $attributes              = $request->all();
            $attributes['user_id']   = user_id();
            $attributes['user_type'] = user_type();
            $call                 = $this->repository->create($attributes);

            return $this->response->message(trans('messages.success.created', ['Module' => trans('database::call.name')]))
                ->code(204)
                ->status('success')
                ->url(guard_url('database/call/' . $call->getRouteKey()))
                ->redirect();
        } catch (Exception $e) {
            return $this->response->message($e->getMessage())
                ->code(400)
                ->status('error')
                ->url(guard_url('/database/call'))
                ->redirect();
        }

    }

    /**
     * Show call for editing.
     *
     * @param Request $request
     * @param Model   $call
     *
     * @return Response
     */
    public function edit(CallRequest $request, Call $call)
    {
        return $this->response->setMetaTitle(trans('app.edit') . ' ' . trans('database::call.name'))
            ->view('database::call.edit', true)
            ->data(compact('call'))
            ->output();
    }

    /**
     * Update the call.
     *
     * @param Request $request
     * @param Model   $call
     *
     * @return Response
     */
    public function update(CallRequest $request, Call $call)
    {
        try {
            $attributes = $request->all();

            $call->update($attributes);
            return $this->response->message(trans('messages.success.updated', ['Module' => trans('database::call.name')]))
                ->code(204)
                ->status('success')
                ->url(guard_url('database/call/' . $call->getRouteKey()))
                ->redirect();
        } catch (Exception $e) {
            return $this->response->message($e->getMessage())
                ->code(400)
                ->status('error')
                ->url(guard_url('database/call/' . $call->getRouteKey()))
                ->redirect();
        }

    }

    /**
     * Remove the call.
     *
     * @param Model   $call
     *
     * @return Response
     */
    public function destroy(CallRequest $request, Call $call)
    {
        try {

            $call->delete();
            return $this->response->message(trans('messages.success.deleted', ['Module' => trans('database::call.name')]))
                ->code(202)
                ->status('success')
                ->url(guard_url('database/call/0'))
                ->redirect();

        } catch (Exception $e) {

            return $this->response->message($e->getMessage())
                ->code(400)
                ->status('error')
                ->url(guard_url('database/call/' . $call->getRouteKey()))
                ->redirect();
        }

    }

    /**
     * Remove multiple call.
     *
     * @param Model   $call
     *
     * @return Response
     */
    public function delete(CallRequest $request, $type)
    {
        try {
            $ids = hashids_decode($request->input('ids'));

            if ($type == 'purge') {
                $this->repository->purge($ids);
            } else {
                $this->repository->delete($ids);
            }

            return $this->response->message(trans('messages.success.deleted', ['Module' => trans('database::call.name')]))
                ->status("success")
                ->code(202)
                ->url(guard_url('database/call'))
                ->redirect();

        } catch (Exception $e) {

            return $this->response->message($e->getMessage())
                ->status("error")
                ->code(400)
                ->url(guard_url('/database/call'))
                ->redirect();
        }

    }

    /**
     * Restore deleted calls.
     *
     * @param Model   $call
     *
     * @return Response
     */
    public function restore(CallRequest $request)
    {
        try {
            $ids = hashids_decode($request->input('ids'));
            $this->repository->restore($ids);

            return $this->response->message(trans('messages.success.restore', ['Module' => trans('database::call.name')]))
                ->status("success")
                ->code(202)
                ->url(guard_url('/database/call'))
                ->redirect();

        } catch (Exception $e) {

            return $this->response->message($e->getMessage())
                ->status("error")
                ->code(400)
                ->url(guard_url('/database/call/'))
                ->redirect();
        }

    }

}
