<?php

namespace Bixo\Database\Http\Controllers;

use App\Http\Controllers\APIController as BaseController;
use Bixo\Database\Http\Requests\DatabaseRequest;
use Bixo\Database\Interfaces\DatabaseRepositoryInterface;
use Bixo\Database\Models\Database;
use Bixo\Database\Forms\Database as Form;

/**
 * APIController  class for database.
 */
class DatabaseAPIController extends BaseController
{

    /**
     * Initialize database resource controller.
     *
     * @param type DatabaseRepositoryInterface $database
     *
     * @return null
     */
    public function __construct(DatabaseRepositoryInterface $database)
    {
        parent::__construct();
        $this->repository = $database;
        $this->repository
            ->pushCriteria(\Litepie\Repository\Criteria\RequestCriteria::class)
            ->pushCriteria(\Bixo\Database\Repositories\Criteria\DatabaseResourceCriteria::class);
    }

    /**
     * Display a list of database.
     *
     * @return Response
     */
    public function index(DatabaseRequest $request)
    {
        return $this->repository
            ->setPresenter(\Bixo\Database\Repositories\Presenter\DatabasePresenter::class)
            ->paginate();
    }

    /**
     * Display database.
     *
     * @param Request $request
     * @param Model   $database
     *
     * @return Response
     */
    public function show(DatabaseRequest $request, Database $database)
    {
        return $database->setPresenter(\Bixo\Database\Repositories\Presenter\DatabaseListPresenter::class);
        ;
    }

    /**
     * Create new database.
     *
     * @param Request $request
     *
     * @return Response
     */
    public function store(DatabaseRequest $request)
    {
        try {
            $data              = $request->all();
            $data['user_id']   = user_id();
            $data['user_type'] = user_type();
            $data              = $this->repository->create($data);
            $message           = trans('messages.success.created', ['Module' => trans('database::database.name')]);
            $code              = 204;
            $status            = 'success';
            $url               = guard_url('database/database/' . $database->getRouteKey());
        } catch (Exception $e) {
            $message = $e->getMessage();
            $code    = 400;
            $status  = 'error';
            $url     = guard_url('database/database');
        }
        return compact('data', 'message', 'code', 'status', 'url');
    }

    /**
     * Update the database.
     *
     * @param Request $request
     * @param Model   $database
     *
     * @return Response
     */
    public function update(DatabaseRequest $request, Database $database)
    {
        try {
            $data = $request->all();

            $database->update($data);
            $message = trans('messages.success.updated', ['Module' => trans('database::database.name')]);
            $code    = 204;
            $status  = 'success';
            $url     = guard_url('database/database/' . $database->getRouteKey());
        } catch (Exception $e) {
            $message = $e->getMessage();
            $code    = 400;
            $status  = 'error';
            $url     = guard_url('database/database/' . $database->getRouteKey());
        }
        return compact('data', 'message', 'code', 'status', 'url');
    }

    /**
     * Remove the database.
     *
     * @param Model   $database
     *
     * @return Response
     */
    public function destroy(DatabaseRequest $request, Database $database)
    {
        try {
            $database->delete();
            $message = trans('messages.success.deleted', ['Module' => trans('database::database.name')]);
            $code    = 202;
            $status  = 'success';
            $url     = guard_url('database/database/0');
        } catch (Exception $e) {
            $message = $e->getMessage();
            $code    = 400;
            $status  = 'error';
            $url     = guard_url('database/database/' . $database->getRouteKey());
        }
        return compact('message', 'code', 'status', 'url');
    }

    /**
     * Return the form elements as json.
     *
     * @param String   $element
     *
     * @return json
     */
    public function form($element = 'fields')
    {
        $form = new Form();
        return $form->form($element, true);
    }

}
