<?php

namespace Bixo\Database\Http\Controllers;

use App\Http\Controllers\ResourceController as BaseController;
use Form;
use Bixo\Database\Http\Requests\DatabaseRequest;
use Bixo\Database\Interfaces\DatabaseRepositoryInterface;
use Bixo\Database\Models\Database;

/**
 * Resource controller class for database.
 */
class DatabaseResourceController extends BaseController
{

    /**
     * Initialize database resource controller.
     *
     * @param type DatabaseRepositoryInterface $database
     *
     * @return null
     */
    public function __construct(DatabaseRepositoryInterface $database)
    {
        parent::__construct();
        $this->repository = $database;
        $this->repository
            ->pushCriteria(\Litepie\Repository\Criteria\RequestCriteria::class)
            ->pushCriteria(\Bixo\Database\Repositories\Criteria\DatabaseResourceCriteria::class);
    }

    /**
     * Display a list of database.
     *
     * @return Response
     */
    public function index(DatabaseRequest $request)
    {
        $view = $this->response->theme->listView();

        if ($this->response->typeIs('json')) {
            $function = camel_case('get-' . $view);
            return $this->repository
                ->setPresenter(\Bixo\Database\Repositories\Presenter\DatabasePresenter::class)
                ->$function();
        }

        $databases = $this->repository->paginate();

        return $this->response->setMetaTitle(trans('database::database.names'))
            ->view('database::database.index', true)
            ->data(compact('databases', 'view'))
            ->output();
    }

    /**
     * Display database.
     *
     * @param Request $request
     * @param Model   $database
     *
     * @return Response
     */
    public function show(DatabaseRequest $request, Database $database)
    {

        if ($database->exists) {
            $view = 'database::database.show';
        } else {
            $view = 'database::database.new';
        }

        return $this->response->setMetaTitle(trans('app.view') . ' ' . trans('database::database.name'))
            ->data(compact('database'))
            ->view($view, true)
            ->output();
    }

    /**
     * Show the form for creating a new database.
     *
     * @param Request $request
     *
     * @return Response
     */
    public function create(DatabaseRequest $request)
    {

        $database = $this->repository->newInstance([]);
        return $this->response->setMetaTitle(trans('app.new') . ' ' . trans('database::database.name')) 
            ->view('database::database.create', true) 
            ->data(compact('database'))
            ->output();
    }

    /**
     * Create new database.
     *
     * @param Request $request
     *
     * @return Response
     */
    public function store(DatabaseRequest $request)
    {
        try {
            $attributes              = $request->all();
            $attributes['user_id']   = user_id();
            $attributes['user_type'] = user_type();
            $database                 = $this->repository->create($attributes);

            return $this->response->message(trans('messages.success.created', ['Module' => trans('database::database.name')]))
                ->code(204)
                ->status('success')
                ->url(guard_url('database/database/' . $database->getRouteKey()))
                ->redirect();
        } catch (Exception $e) {
            return $this->response->message($e->getMessage())
                ->code(400)
                ->status('error')
                ->url(guard_url('/database/database'))
                ->redirect();
        }

    }

    /**
     * Show database for editing.
     *
     * @param Request $request
     * @param Model   $database
     *
     * @return Response
     */
    public function edit(DatabaseRequest $request, Database $database)
    {
        return $this->response->setMetaTitle(trans('app.edit') . ' ' . trans('database::database.name'))
            ->view('database::database.edit', true)
            ->data(compact('database'))
            ->output();
    }

    /**
     * Update the database.
     *
     * @param Request $request
     * @param Model   $database
     *
     * @return Response
     */
    public function update(DatabaseRequest $request, Database $database)
    {
        try {
            $attributes = $request->all();

            $database->update($attributes);
            return $this->response->message(trans('messages.success.updated', ['Module' => trans('database::database.name')]))
                ->code(204)
                ->status('success')
                ->url(guard_url('database/database/' . $database->getRouteKey()))
                ->redirect();
        } catch (Exception $e) {
            return $this->response->message($e->getMessage())
                ->code(400)
                ->status('error')
                ->url(guard_url('database/database/' . $database->getRouteKey()))
                ->redirect();
        }

    }

    /**
     * Remove the database.
     *
     * @param Model   $database
     *
     * @return Response
     */
    public function destroy(DatabaseRequest $request, Database $database)
    {
        try {

            $database->delete();
            return $this->response->message(trans('messages.success.deleted', ['Module' => trans('database::database.name')]))
                ->code(202)
                ->status('success')
                ->url(guard_url('database/database/0'))
                ->redirect();

        } catch (Exception $e) {

            return $this->response->message($e->getMessage())
                ->code(400)
                ->status('error')
                ->url(guard_url('database/database/' . $database->getRouteKey()))
                ->redirect();
        }

    }

    /**
     * Remove multiple database.
     *
     * @param Model   $database
     *
     * @return Response
     */
    public function delete(DatabaseRequest $request, $type)
    {
        try {
            $ids = hashids_decode($request->input('ids'));

            if ($type == 'purge') {
                $this->repository->purge($ids);
            } else {
                $this->repository->delete($ids);
            }

            return $this->response->message(trans('messages.success.deleted', ['Module' => trans('database::database.name')]))
                ->status("success")
                ->code(202)
                ->url(guard_url('database/database'))
                ->redirect();

        } catch (Exception $e) {

            return $this->response->message($e->getMessage())
                ->status("error")
                ->code(400)
                ->url(guard_url('/database/database'))
                ->redirect();
        }

    }

    /**
     * Restore deleted databases.
     *
     * @param Model   $database
     *
     * @return Response
     */
    public function restore(DatabaseRequest $request)
    {
        try {
            $ids = hashids_decode($request->input('ids'));
            $this->repository->restore($ids);

            return $this->response->message(trans('messages.success.restore', ['Module' => trans('database::database.name')]))
                ->status("success")
                ->code(202)
                ->url(guard_url('/database/database'))
                ->redirect();

        } catch (Exception $e) {

            return $this->response->message($e->getMessage())
                ->status("error")
                ->code(400)
                ->url(guard_url('/database/database/'))
                ->redirect();
        }

    }

}
