<?php

namespace Bixo\Database\Http\Controllers;

use App\Http\Controllers\APIController as BaseController;
use Bixo\Database\Http\Requests\UploadRequest;
use Bixo\Database\Interfaces\UploadRepositoryInterface;
use Bixo\Database\Models\Upload;
use Bixo\Database\Forms\Upload as Form;

/**
 * APIController  class for upload.
 */
class UploadAPIController extends BaseController
{

    /**
     * Initialize upload resource controller.
     *
     * @param type UploadRepositoryInterface $upload
     *
     * @return null
     */
    public function __construct(UploadRepositoryInterface $upload)
    {
        parent::__construct();
        $this->repository = $upload;
        $this->repository
            ->pushCriteria(\Litepie\Repository\Criteria\RequestCriteria::class)
            ->pushCriteria(\Bixo\Database\Repositories\Criteria\UploadResourceCriteria::class);
    }

    /**
     * Display a list of upload.
     *
     * @return Response
     */
    public function index(UploadRequest $request)
    {
        return $this->repository
            ->setPresenter(\Bixo\Database\Repositories\Presenter\UploadPresenter::class)
            ->paginate();
    }

    /**
     * Display upload.
     *
     * @param Request $request
     * @param Model   $upload
     *
     * @return Response
     */
    public function show(UploadRequest $request, Upload $upload)
    {
        return $upload->setPresenter(\Bixo\Database\Repositories\Presenter\UploadListPresenter::class);
        ;
    }

    /**
     * Create new upload.
     *
     * @param Request $request
     *
     * @return Response
     */
    public function store(UploadRequest $request)
    {
        try {
            $data              = $request->all();
            $data['user_id']   = user_id();
            $data['user_type'] = user_type();
            $data              = $this->repository->create($data);
            $message           = trans('messages.success.created', ['Module' => trans('database::upload.name')]);
            $code              = 204;
            $status            = 'success';
            $url               = guard_url('database/upload/' . $upload->getRouteKey());
        } catch (Exception $e) {
            $message = $e->getMessage();
            $code    = 400;
            $status  = 'error';
            $url     = guard_url('database/upload');
        }
        return compact('data', 'message', 'code', 'status', 'url');
    }

    /**
     * Update the upload.
     *
     * @param Request $request
     * @param Model   $upload
     *
     * @return Response
     */
    public function update(UploadRequest $request, Upload $upload)
    {
        try {
            $data = $request->all();

            $upload->update($data);
            $message = trans('messages.success.updated', ['Module' => trans('database::upload.name')]);
            $code    = 204;
            $status  = 'success';
            $url     = guard_url('database/upload/' . $upload->getRouteKey());
        } catch (Exception $e) {
            $message = $e->getMessage();
            $code    = 400;
            $status  = 'error';
            $url     = guard_url('database/upload/' . $upload->getRouteKey());
        }
        return compact('data', 'message', 'code', 'status', 'url');
    }

    /**
     * Remove the upload.
     *
     * @param Model   $upload
     *
     * @return Response
     */
    public function destroy(UploadRequest $request, Upload $upload)
    {
        try {
            $upload->delete();
            $message = trans('messages.success.deleted', ['Module' => trans('database::upload.name')]);
            $code    = 202;
            $status  = 'success';
            $url     = guard_url('database/upload/0');
        } catch (Exception $e) {
            $message = $e->getMessage();
            $code    = 400;
            $status  = 'error';
            $url     = guard_url('database/upload/' . $upload->getRouteKey());
        }
        return compact('message', 'code', 'status', 'url');
    }

    /**
     * Return the form elements as json.
     *
     * @param String   $element
     *
     * @return json
     */
    public function form($element = 'fields')
    {
        $form = new Form();
        return $form->form($element, true);
    }

}
