<?php

namespace Bixo\Database\Http\Controllers;

use App\Http\Controllers\ResourceController as BaseController;
use Form;
use Bixo\Database\Http\Requests\UploadRequest;
use Bixo\Database\Interfaces\UploadRepositoryInterface;
use Bixo\Database\Models\Upload;

/**
 * Resource controller class for upload.
 */
class UploadResourceController extends BaseController
{

    /**
     * Initialize upload resource controller.
     *
     * @param type UploadRepositoryInterface $upload
     *
     * @return null
     */
    public function __construct(UploadRepositoryInterface $upload)
    {
        parent::__construct();
        $this->repository = $upload;
        $this->repository
            ->pushCriteria(\Litepie\Repository\Criteria\RequestCriteria::class)
            ->pushCriteria(\Bixo\Database\Repositories\Criteria\UploadResourceCriteria::class);
    }

    /**
     * Display a list of upload.
     *
     * @return Response
     */
    public function index(UploadRequest $request)
    {
        $view = $this->response->theme->listView();

        if ($this->response->typeIs('json')) {
            $function = camel_case('get-' . $view);
            return $this->repository
                ->setPresenter(\Bixo\Database\Repositories\Presenter\UploadPresenter::class)
                ->$function();
        }

        $uploads = $this->repository->paginate();

        return $this->response->setMetaTitle(trans('database::upload.names'))
            ->view('database::upload.index', true)
            ->data(compact('uploads', 'view'))
            ->output();
    }

    /**
     * Display upload.
     *
     * @param Request $request
     * @param Model   $upload
     *
     * @return Response
     */
    public function show(UploadRequest $request, Upload $upload)
    {

        if ($upload->exists) {
            $view = 'database::upload.show';
        } else {
            $view = 'database::upload.new';
        }

        return $this->response->setMetaTitle(trans('app.view') . ' ' . trans('database::upload.name'))
            ->data(compact('upload'))
            ->view($view, true)
            ->output();
    }

    /**
     * Show the form for creating a new upload.
     *
     * @param Request $request
     *
     * @return Response
     */
    public function create(UploadRequest $request)
    {

        $upload = $this->repository->newInstance([]);
        return $this->response->setMetaTitle(trans('app.new') . ' ' . trans('database::upload.name')) 
            ->view('database::upload.create', true) 
            ->data(compact('upload'))
            ->output();
    }

    /**
     * Create new upload.
     *
     * @param Request $request
     *
     * @return Response
     */
    public function store(UploadRequest $request)
    {
        try {
            $attributes              = $request->all();
            $attributes['user_id']   = user_id();
            $attributes['user_type'] = user_type();
            $upload                 = $this->repository->create($attributes);

            return $this->response->message(trans('messages.success.created', ['Module' => trans('database::upload.name')]))
                ->code(204)
                ->status('success')
                ->url(guard_url('database/upload/' . $upload->getRouteKey()))
                ->redirect();
        } catch (Exception $e) {
            return $this->response->message($e->getMessage())
                ->code(400)
                ->status('error')
                ->url(guard_url('/database/upload'))
                ->redirect();
        }

    }

    /**
     * Show upload for editing.
     *
     * @param Request $request
     * @param Model   $upload
     *
     * @return Response
     */
    public function edit(UploadRequest $request, Upload $upload)
    {
        return $this->response->setMetaTitle(trans('app.edit') . ' ' . trans('database::upload.name'))
            ->view('database::upload.edit', true)
            ->data(compact('upload'))
            ->output();
    }

    /**
     * Update the upload.
     *
     * @param Request $request
     * @param Model   $upload
     *
     * @return Response
     */
    public function update(UploadRequest $request, Upload $upload)
    {
        try {
            $attributes = $request->all();

            $upload->update($attributes);
            return $this->response->message(trans('messages.success.updated', ['Module' => trans('database::upload.name')]))
                ->code(204)
                ->status('success')
                ->url(guard_url('database/upload/' . $upload->getRouteKey()))
                ->redirect();
        } catch (Exception $e) {
            return $this->response->message($e->getMessage())
                ->code(400)
                ->status('error')
                ->url(guard_url('database/upload/' . $upload->getRouteKey()))
                ->redirect();
        }

    }

    /**
     * Remove the upload.
     *
     * @param Model   $upload
     *
     * @return Response
     */
    public function destroy(UploadRequest $request, Upload $upload)
    {
        try {

            $upload->delete();
            return $this->response->message(trans('messages.success.deleted', ['Module' => trans('database::upload.name')]))
                ->code(202)
                ->status('success')
                ->url(guard_url('database/upload/0'))
                ->redirect();

        } catch (Exception $e) {

            return $this->response->message($e->getMessage())
                ->code(400)
                ->status('error')
                ->url(guard_url('database/upload/' . $upload->getRouteKey()))
                ->redirect();
        }

    }

    /**
     * Remove multiple upload.
     *
     * @param Model   $upload
     *
     * @return Response
     */
    public function delete(UploadRequest $request, $type)
    {
        try {
            $ids = hashids_decode($request->input('ids'));

            if ($type == 'purge') {
                $this->repository->purge($ids);
            } else {
                $this->repository->delete($ids);
            }

            return $this->response->message(trans('messages.success.deleted', ['Module' => trans('database::upload.name')]))
                ->status("success")
                ->code(202)
                ->url(guard_url('database/upload'))
                ->redirect();

        } catch (Exception $e) {

            return $this->response->message($e->getMessage())
                ->status("error")
                ->code(400)
                ->url(guard_url('/database/upload'))
                ->redirect();
        }

    }

    /**
     * Restore deleted uploads.
     *
     * @param Model   $upload
     *
     * @return Response
     */
    public function restore(UploadRequest $request)
    {
        try {
            $ids = hashids_decode($request->input('ids'));
            $this->repository->restore($ids);

            return $this->response->message(trans('messages.success.restore', ['Module' => trans('database::upload.name')]))
                ->status("success")
                ->code(202)
                ->url(guard_url('/database/upload'))
                ->redirect();

        } catch (Exception $e) {

            return $this->response->message($e->getMessage())
                ->status("error")
                ->code(400)
                ->url(guard_url('/database/upload/'))
                ->redirect();
        }

    }

}
