<?php

namespace Bixo\Developer\Http\Controllers;

use Exception;
use Litepie\Http\Controllers\ResourceController as BaseController;
use Bixo\Developer\Forms\Developer as DeveloperForm;
use Bixo\Developer\Http\Requests\DeveloperResourceRequest;
use Bixo\Developer\Http\Resources\DeveloperResource;
use Bixo\Developer\Http\Resources\DevelopersCollection;
use Bixo\Developer\Models\Developer;
use Bixo\Developer\Actions\DeveloperAction;
use Bixo\Developer\Actions\DeveloperActions;


/**
 * Resource controller class for developer.
 */
class DeveloperResourceController extends BaseController
{

    /**
     * Initialize developer resource controller.
     *
     *
     * @return null
     */
    public function __construct()
    {
        parent::__construct();
        $this->middleware(function ($request, $next) {
            $this->form = DeveloperForm::grouped(false)
                            ->setAttributes()
                            ->toArray();
            $this->modules = $this->modules(config('bixo.developer.modules'), 'developer', guard_url('developer'));
            return $next($request);
        });
    }

    /**
     * Display a list of developer.
     *
     * @return Response
     */
    public function index(DeveloperResourceRequest $request)
    {
        $request = $request->all();
        $page = DeveloperActions::run('paginate', $request);

        $data = new DevelopersCollection($page);

        $form = $this->form;
        $modules = $this->modules;

        return $this->response->setMetaTitle(trans('developer::developer.names'))
            ->view('developer::developer.index')
            ->data(compact('data', 'modules', 'form'))
            ->output();

    }

    /**
     * Display developer.
     *
     * @param Request $request
     * @param Model   $developer
     *
     * @return Response
     */
    public function show(DeveloperResourceRequest $request, Developer $model)
    {
        $form = $this->form;
        $modules = $this->modules;
        $data = new DeveloperResource($model);
        return $this->response
            ->setMetaTitle(trans('app.view') . ' ' . trans('developer::developer.name'))
            ->data(compact('data', 'form', 'modules'))
            ->view('developer::developer.show')
            ->output();
    }

    /**
     * Show the form for creating a new developer.
     *
     * @param Request $request
     *
     * @return Response
     */
    public function create(DeveloperResourceRequest $request, Developer $model)
    {
        $form = $this->form;
        $modules = $this->modules;
        $data = new DeveloperResource($model);
        return $this->response->setMetaTitle(trans('app.new') . ' ' . trans('developer::developer.name'))
            ->view('developer::developer.create')
            ->data(compact('data', 'form', 'modules'))
            ->output();

    }

    /**
     * Create new developer.
     *
     * @param Request $request
     *
     * @return Response
     */
    public function store(DeveloperResourceRequest $request, Developer $model)
    {
        try {
            $request = $request->all();
            $model = DeveloperAction::run('store', $model, $request);
            $data = new DeveloperResource($model);
            return $this->response->message(trans('messages.success.created', ['Module' => trans('developer::developer.name')]))
                ->code(204)
                ->data(compact('data'))
                ->status('success')
                ->url(guard_url('developer/developer/' . $model->getRouteKey()))
                ->redirect();
        } catch (Exception $e) {
            return $this->response->message($e->getMessage())
                ->code(400)
                ->status('error')
                ->url(guard_url('/developer/developer'))
                ->redirect();
        }

    }

    /**
     * Show developer for editing.
     *
     * @param Request $request
     * @param Model   $developer
     *
     * @return Response
     */
    public function edit(DeveloperResourceRequest $request, Developer $model)
    {
        $form = $this->form;
        $modules = $this->modules;
        $data = new DeveloperResource($model);
        // return view('developer::developer.edit', compact('data', 'form', 'modules'));

        return $this->response->setMetaTitle(trans('app.edit') . ' ' . trans('developer::developer.name'))
            ->view('developer::developer.edit')
            ->data(compact('data', 'form', 'modules'))
            ->output();

    }

    /**
     * Update the developer.
     *
     * @param Request $request
     * @param Model   $developer
     *
     * @return Response
     */
    public function update(DeveloperResourceRequest $request, Developer $model)
    {
        try {
            $request = $request->all();
            $model = DeveloperAction::run('update', $model, $request);
            $data = new DeveloperResource($model);

            return $this->response->message(trans('messages.success.updated', ['Module' => trans('developer::developer.name')]))
                ->code(204)
                ->status('success')
                ->data(compact('data'))
                ->url(guard_url('developer/developer/' . $model->getRouteKey()))
                ->redirect();
        } catch (Exception $e) {
            return $this->response->message($e->getMessage())
                ->code(400)
                ->status('error')
                ->url(guard_url('developer/developer/' .  $model->getRouteKey()))
                ->redirect();
        }

    }

    /**
     * Remove the developer.
     *
     * @param Model   $developer
     *
     * @return Response
     */
    public function destroy(DeveloperResourceRequest $request, Developer $model)
    {
        try {

            $request = $request->all();
            $model = DeveloperAction::run('destroy', $model, $request);
            $data = new DeveloperResource($model);

            return $this->response->message(trans('messages.success.deleted', ['Module' => trans('developer::developer.name')]))
                ->code(202)
                ->status('success')
                ->data(compact('data'))
                ->url(guard_url('developer/developer/0'))
                ->redirect();

        } catch (Exception $e) {

            return $this->response->message($e->getMessage())
                ->code(400)
                ->status('error')
                ->url(guard_url('developer/developer/' .  $model->getRouteKey()))
                ->redirect();
        }

    }
}
