<?php

namespace Bixo\Developer\Http\Controllers;

use Exception;
use Bixo\Developer\Actions\DeveloperWorkflow;
use Bixo\Developer\Http\Requests\DeveloperWorkflowRequest;
use Bixo\Developer\Http\Resources\DeveloperResource;
use Bixo\Developer\Models\Developer;
use Litepie\Http\Controllers\ActionController as BaseController;

/**
 * Workflow controller class.
 *
 */
class DeveloperWorkflowController extends BaseController
{
    /**
     * Action controller function for developer.
     *
     * @param Model $developer
     * @param action next action for the developer.
     *
     * @return Response
     */
    public function __invoke(DeveloperWorkflowRequest $request, Developer $developer, $trasition)
    {
        try {
            $request = $request->all();
            $developer = DeveloperWorkflow::run($trasition, $developer, $request);
            $data = new DeveloperResource($developer);
            return $this->response->message(trans('messages.success.updated', ['Module' => trans('developer::developer.name')]))
                ->code(204)
                ->data(compact('data'))
                ->status('success')
                ->url(guard_url('developer/developer/' . $developer->getRouteKey()))
                ->redirect();
        } catch (Exception $e) {
            return $this->response->message($e->getMessage())
                ->code(400)
                ->status('error')
                ->url(guard_url('developer/developer/' . $developer->getRouteKey()))
                ->redirect();
        }
    }


    /**
     * Get the developer and user for the workflow.
     *
     * @param PublicRequest $request
     * @param string $developer
     * @param string $user
     *
     * @return Response
     */
    public function get(PublicRequest $request, $developer, $user)
    {
        if (!$request->hasValidSignature()) {
            abort(403, 'This url is expired!');
        }

        $auth = Auth::guard()->getProvider()->getModel();
        $user = $auth::findBySignedId($user);
        Auth::guard()->login($user);
        $developer = Developer::findBySignedId($developer);
        $transitions = $this->transitions($developer, $user);
        $user = user();
        Auth::logout();
        return $this->response->setMetaTitle(trans('developer::developer.name') . ' ' . trans('Approvals'))
            ->view('developer::public.developer.workflow')
            ->layout('mobile')
            ->data(compact('developer', 'user', 'transitions'))
            ->output();
    }

    /**
     * Display the workflow form for the developer.
     *
     * @param PublicRequest $request
     * @param string $developer
     * @param string $user
     *
     * @return Response
     */
    public function post(PublicRequest $request, $developer, $user)
    {
        if (!$request->hasValidSignature()) {
            abort(403, 'This url is expired!');
        }
        $transition = $request->transition;
        $auth = Auth::guard()->getProvider()->getModel();
        $user = $auth::findBySignedId($user);
        Auth::guard()->login($user);
        $developer = Developer::findBySignedId($developer);
        $user = user();
        $developer = DeveloperWorkflow::run($transition, $developer, $request->all());
        Auth::logout();

        return response()->json(
            [
                'status' => 'success',
                'url' => $request->url(),
            ]
        );
    }

    private function transitions($developer, $user)
    {
        $transitions = [];
        foreach ($developer->workflow()->transitions($developer) as $key => $value) {
            $name = $value->getName();
            $array['url'] = URL::temporarySignedRoute('litecms.developer.workflow', now()->addMinutes(3), [
                'guard' => 'admin',
                'transition' => $name,
                'developer' => $developer->getSignedId(now()->addMinutes(3)),
                'user' => $user->getSignedId(now()->addMinutes(3)),
                'trans' => 'en',
            ]
            );
            $array['name'] = $name;
            $array['key'] = $name;
            $array['form'] = $developer->workflow()->form($value);
            $array['label'] = trans('developer::developer.workflow.' . $name);
            $transitions[] = $array;
        }
        return $transitions;
    }
}
