<?php

namespace Superrobota\Document\Http\Controllers;

use App\Http\Controllers\ResourceController as BaseController;
use Form;
use Superrobota\Document\Http\Requests\DocumentsRequest;
use Superrobota\Document\Interfaces\DocumentsRepositoryInterface;
use Superrobota\Document\Models\Documents;

/**
 * Resource controller class for documents.
 */
class DocumentsResourceController extends BaseController
{

    /**
     * Initialize documents resource controller.
     *
     * @param type DocumentsRepositoryInterface $documents
     *
     * @return null
     */
    public function __construct(DocumentsRepositoryInterface $documents)
    {
        parent::__construct();
        $this->repository = $documents;
        $this->repository
            ->pushCriteria(\Litepie\Repository\Criteria\RequestCriteria::class)
            ->pushCriteria(\Superrobota\Document\Repositories\Criteria\DocumentsResourceCriteria::class);
    }

    /**
     * Display a list of documents.
     *
     * @return Response
     */
    public function index(DocumentsRequest $request)
    {
        $view = $this->response->theme->listView();

        if ($this->response->typeIs('json')) {
            $function = camel_case('get-' . $view);
            return $this->repository
                ->setPresenter(\Superrobota\Document\Repositories\Presenter\DocumentsPresenter::class)
                ->$function();
        }

        $documents = $this->repository->paginate();

        return $this->response->title(trans('document::documents.names'))
            ->view('document::documents.index', true)
            ->data(compact('documents'))
            ->output();
    }

    /**
     * Display documents.
     *
     * @param Request $request
     * @param Model   $documents
     *
     * @return Response
     */
    public function show(DocumentsRequest $request, Documents $documents)
    {

        if ($documents->exists) {
            $view = 'document::documents.show';
        } else {
            $view = 'document::documents.new';
        }

        return $this->response->title(trans('app.view') . ' ' . trans('document::documents.name'))
            ->data(compact('documents'))
            ->view($view, true)
            ->output();
    }

    /**
     * Show the form for creating a new documents.
     *
     * @param Request $request
     *
     * @return Response
     */
    public function create(DocumentsRequest $request)
    {

        $documents = $this->repository->newInstance([]);
        return $this->response->title(trans('app.new') . ' ' . trans('document::documents.name')) 
            ->view('document::documents.create', true) 
            ->data(compact('documents'))
            ->output();
    }

    /**
     * Create new documents.
     *
     * @param Request $request
     *
     * @return Response
     */
    public function store(DocumentsRequest $request)
    {
        try {
            $attributes              = $request->all();
            $attributes['user_id']   = user_id();
            $attributes['user_type'] = user_type();
            $documents                 = $this->repository->create($attributes);

            return $this->response->message(trans('messages.success.created', ['Module' => trans('document::documents.name')]))
                ->code(204)
                ->status('success')
                ->url(guard_url('document/documents/' . $documents->getRouteKey()))
                ->redirect();
        } catch (Exception $e) {
            return $this->response->message($e->getMessage())
                ->code(400)
                ->status('error')
                ->url(guard_url('/document/documents'))
                ->redirect();
        }

    }

    /**
     * Show documents for editing.
     *
     * @param Request $request
     * @param Model   $documents
     *
     * @return Response
     */
    public function edit(DocumentsRequest $request, Documents $documents)
    {
        return $this->response->title(trans('app.edit') . ' ' . trans('document::documents.name'))
            ->view('document::documents.edit', true)
            ->data(compact('documents'))
            ->output();
    }

    /**
     * Update the documents.
     *
     * @param Request $request
     * @param Model   $documents
     *
     * @return Response
     */
    public function update(DocumentsRequest $request, Documents $documents)
    {
        try {
            $attributes = $request->all();

            $documents->update($attributes);
            return $this->response->message(trans('messages.success.updated', ['Module' => trans('document::documents.name')]))
                ->code(204)
                ->status('success')
                ->url(guard_url('document/documents/' . $documents->getRouteKey()))
                ->redirect();
        } catch (Exception $e) {
            return $this->response->message($e->getMessage())
                ->code(400)
                ->status('error')
                ->url(guard_url('document/documents/' . $documents->getRouteKey()))
                ->redirect();
        }

    }

    /**
     * Remove the documents.
     *
     * @param Model   $documents
     *
     * @return Response
     */
    public function destroy(DocumentsRequest $request, Documents $documents)
    {
        try {

            $documents->delete();
            return $this->response->message(trans('messages.success.deleted', ['Module' => trans('document::documents.name')]))
                ->code(202)
                ->status('success')
                ->url(guard_url('document/documents/0'))
                ->redirect();

        } catch (Exception $e) {

            return $this->response->message($e->getMessage())
                ->code(400)
                ->status('error')
                ->url(guard_url('document/documents/' . $documents->getRouteKey()))
                ->redirect();
        }

    }

    /**
     * Remove multiple documents.
     *
     * @param Model   $documents
     *
     * @return Response
     */
    public function delete(DocumentsRequest $request, $type)
    {
        try {
            $ids = hashids_decode($request->input('ids'));

            if ($type == 'purge') {
                $this->repository->purge($ids);
            } else {
                $this->repository->delete($ids);
            }

            return $this->response->message(trans('messages.success.deleted', ['Module' => trans('document::documents.name')]))
                ->status("success")
                ->code(202)
                ->url(guard_url('document/documents'))
                ->redirect();

        } catch (Exception $e) {

            return $this->response->message($e->getMessage())
                ->status("error")
                ->code(400)
                ->url(guard_url('/document/documents'))
                ->redirect();
        }

    }

    /**
     * Restore deleted documents.
     *
     * @param Model   $documents
     *
     * @return Response
     */
    public function restore(DocumentsRequest $request)
    {
        try {
            $ids = hashids_decode($request->input('ids'));
            $this->repository->restore($ids);

            return $this->response->message(trans('messages.success.restore', ['Module' => trans('document::documents.name')]))
                ->status("success")
                ->code(202)
                ->url(guard_url('/document/documents'))
                ->redirect();

        } catch (Exception $e) {

            return $this->response->message($e->getMessage())
                ->status("error")
                ->code(400)
                ->url(guard_url('/document/documents/'))
                ->redirect();
        }

    }

}
