<?php

namespace Test\Test\Http\Controllers;

use App\Http\Controllers\AdminWebController as AdminController;
use Form;
use Test\Test\Http\Requests\NewsAdminWebRequest;
use Test\Test\Interfaces\NewsRepositoryInterface;
use Test\Test\Models\News;

/**
 * Admin web controller class.
 */
class NewsAdminWebController extends AdminController
{
    /**
     * Initialize news controller.
     *
     * @param type NewsRepositoryInterface $news
     *
     * @return type
     */
    public function __construct(NewsRepositoryInterface $news)
    {
        $this->repository = $news;
        parent::__construct();
    }

    /**
     * Display a list of news.
     *
     * @return Response
     */
    public function index(NewsAdminWebRequest $request)
    {
        if ($request->wantsJson()) {
            $news  = $this->repository->setPresenter('\\Test\\Test\\Repositories\\Presenter\\NewsListPresenter')
                                                ->scopeQuery(function($query){
                                                    return $query->orderBy('id','DESC');
                                                })->all();
            return response()->json($news, 200);

        }
        $this   ->theme->prependTitle(trans('test::news.names').' :: ');
        return $this->theme->of('test::admin.news.index')->render();
    }

    /**
     * Display news.
     *
     * @param Request $request
     * @param Model   $news
     *
     * @return Response
     */
    public function show(NewsAdminWebRequest $request, News $news)
    {
        if (!$news->exists) {
            return response()->view('test::admin.news.new', compact('news'));
        }

        Form::populate($news);
        return response()->view('test::admin.news.show', compact('news'));
    }

    /**
     * Show the form for creating a new news.
     *
     * @param Request $request
     *
     * @return Response
     */
    public function create(NewsAdminWebRequest $request)
    {

        $news = $this->repository->newInstance([]);

        Form::populate($news);

        return response()->view('test::admin.news.create', compact('news'));

    }

    /**
     * Create new news.
     *
     * @param Request $request
     *
     * @return Response
     */
    public function store(NewsAdminWebRequest $request)
    {
        try {
            $attributes             = $request->all();
            $attributes['user_id']  = user_id('admin.web');
            $news          = $this->repository->create($attributes);

            return response()->json([
                'message'  => trans('messages.success.updated', ['Module' => trans('test::news.name')]),
                'code'     => 204,
                'redirect' => trans_url('/admin/test/news/'.$news->getRouteKey())
            ], 201);


        } catch (Exception $e) {
            return response()->json([
                'message'  => $e->getMessage(),
                'code'     => 400,
            ], 400);
        }
    }

    /**
     * Show news for editing.
     *
     * @param Request $request
     * @param Model   $news
     *
     * @return Response
     */
    public function edit(NewsAdminWebRequest $request, News $news)
    {
        Form::populate($news);
        return  response()->view('test::admin.news.edit', compact('news'));
    }

    /**
     * Update the news.
     *
     * @param Request $request
     * @param Model   $news
     *
     * @return Response
     */
    public function update(NewsAdminWebRequest $request, News $news)
    {
        try {

            $attributes = $request->all();

            $news->update($attributes);

            return response()->json([
                'message'  => trans('messages.success.updated', ['Module' => trans('test::news.name')]),
                'code'     => 204,
                'redirect' => trans_url('/admin/test/news/'.$news->getRouteKey())
            ], 201);

        } catch (Exception $e) {

            return response()->json([
                'message'  => $e->getMessage(),
                'code'     => 400,
                'redirect' => trans_url('/admin/test/news/'.$news->getRouteKey()),
            ], 400);

        }
    }

    /**
     * Remove the news.
     *
     * @param Model   $news
     *
     * @return Response
     */
    public function destroy(NewsAdminWebRequest $request, News $news)
    {

        try {

            $t = $news->delete();

            return response()->json([
                'message'  => trans('messages.success.deleted', ['Module' => trans('test::news.name')]),
                'code'     => 202,
                'redirect' => trans_url('/admin/test/news/0'),
            ], 202);

        } catch (Exception $e) {

            return response()->json([
                'message'  => $e->getMessage(),
                'code'     => 400,
                'redirect' => trans_url('/admin/test/news/'.$news->getRouteKey()),
            ], 400);
        }
    }
}
