<?php

namespace Example\Example\Http\Controllers;

use App\Http\Controllers\APIController as BaseController;
use Example\Example\Http\Requests\ExampleRequest;
use Example\Example\Interfaces\ExampleRepositoryInterface;
use Example\Example\Models\Example;
use Example\Example\Forms\Example as Form;

/**
 * APIController  class for example.
 */
class ExampleAPIController extends BaseController
{

    /**
     * Initialize example resource controller.
     *
     * @param type ExampleRepositoryInterface $example
     *
     * @return null
     */
    public function __construct(ExampleRepositoryInterface $example)
    {
        parent::__construct();
        $this->repository = $example;
        $this->repository
            ->pushCriteria(\Litepie\Repository\Criteria\RequestCriteria::class)
            ->pushCriteria(\Example\Example\Repositories\Criteria\ExampleResourceCriteria::class);
    }

    /**
     * Display a list of example.
     *
     * @return Response
     */
    public function index(ExampleRequest $request)
    {
        return $this->repository
            ->setPresenter(\Example\Example\Repositories\Presenter\ExamplePresenter::class)
            ->paginate();
    }

    /**
     * Display example.
     *
     * @param Request $request
     * @param Model   $example
     *
     * @return Response
     */
    public function show(ExampleRequest $request, Example $example)
    {
        return $example->setPresenter(\Example\Example\Repositories\Presenter\ExampleListPresenter::class);
        ;
    }

    /**
     * Create new example.
     *
     * @param Request $request
     *
     * @return Response
     */
    public function store(ExampleRequest $request)
    {
        try {
            $data              = $request->all();
            $data['user_id']   = user_id();
            $data['user_type'] = user_type();
            $data              = $this->repository->create($data);
            $message           = trans('messages.success.created', ['Module' => trans('example::example.name')]);
            $code              = 204;
            $status            = 'success';
            $url               = guard_url('example/example/' . $example->getRouteKey());
        } catch (Exception $e) {
            $message = $e->getMessage();
            $code    = 400;
            $status  = 'error';
            $url     = guard_url('example/example');
        }
        return compact('data', 'message', 'code', 'status', 'url');
    }

    /**
     * Update the example.
     *
     * @param Request $request
     * @param Model   $example
     *
     * @return Response
     */
    public function update(ExampleRequest $request, Example $example)
    {
        try {
            $data = $request->all();

            $example->update($data);
            $message = trans('messages.success.updated', ['Module' => trans('example::example.name')]);
            $code    = 204;
            $status  = 'success';
            $url     = guard_url('example/example/' . $example->getRouteKey());
        } catch (Exception $e) {
            $message = $e->getMessage();
            $code    = 400;
            $status  = 'error';
            $url     = guard_url('example/example/' . $example->getRouteKey());
        }
        return compact('data', 'message', 'code', 'status', 'url');
    }

    /**
     * Remove the example.
     *
     * @param Model   $example
     *
     * @return Response
     */
    public function destroy(ExampleRequest $request, Example $example)
    {
        try {
            $example->delete();
            $message = trans('messages.success.deleted', ['Module' => trans('example::example.name')]);
            $code    = 202;
            $status  = 'success';
            $url     = guard_url('example/example/0');
        } catch (Exception $e) {
            $message = $e->getMessage();
            $code    = 400;
            $status  = 'error';
            $url     = guard_url('example/example/' . $example->getRouteKey());
        }
        return compact('message', 'code', 'status', 'url');
    }

    /**
     * Return the form elements as json.
     *
     * @param String   $element
     *
     * @return json
     */
    public function form($element = 'fields')
    {
        $form = new Form();
        return $form->form($element, true);
    }

}
