<?php

namespace B2buy\Help\Http\Controllers;

use App\Http\Controllers\APIController as BaseController;
use B2buy\Help\Http\Requests\HelpRequest;
use B2buy\Help\Interfaces\HelpRepositoryInterface;
use B2buy\Help\Models\Help;
use B2buy\Help\Forms\Help as Form;

/**
 * APIController  class for help.
 */
class HelpAPIController extends BaseController
{

    /**
     * Initialize help resource controller.
     *
     * @param type HelpRepositoryInterface $help
     *
     * @return null
     */
    public function __construct(HelpRepositoryInterface $help)
    {
        parent::__construct();
        $this->repository = $help;
        $this->repository
            ->pushCriteria(\Litepie\Repository\Criteria\RequestCriteria::class)
            ->pushCriteria(\B2buy\Help\Repositories\Criteria\HelpResourceCriteria::class);
    }

    /**
     * Display a list of help.
     *
     * @return Response
     */
    public function index(HelpRequest $request)
    {
        return $this->repository
            ->setPresenter(\B2buy\Help\Repositories\Presenter\HelpPresenter::class)
            ->paginate();
    }

    /**
     * Display help.
     *
     * @param Request $request
     * @param Model   $help
     *
     * @return Response
     */
    public function show(HelpRequest $request, Help $help)
    {
        return $help->setPresenter(\B2buy\Help\Repositories\Presenter\HelpListPresenter::class);
        ;
    }

    /**
     * Create new help.
     *
     * @param Request $request
     *
     * @return Response
     */
    public function store(HelpRequest $request)
    {
        try {
            $data              = $request->all();
            $data['user_id']   = user_id();
            $data['user_type'] = user_type();
            $data              = $this->repository->create($data);
            $message           = trans('messages.success.created', ['Module' => trans('help::help.name')]);
            $code              = 204;
            $status            = 'success';
            $url               = guard_url('help/help/' . $help->getRouteKey());
        } catch (Exception $e) {
            $message = $e->getMessage();
            $code    = 400;
            $status  = 'error';
            $url     = guard_url('help/help');
        }
        return compact('data', 'message', 'code', 'status', 'url');
    }

    /**
     * Update the help.
     *
     * @param Request $request
     * @param Model   $help
     *
     * @return Response
     */
    public function update(HelpRequest $request, Help $help)
    {
        try {
            $data = $request->all();

            $help->update($data);
            $message = trans('messages.success.updated', ['Module' => trans('help::help.name')]);
            $code    = 204;
            $status  = 'success';
            $url     = guard_url('help/help/' . $help->getRouteKey());
        } catch (Exception $e) {
            $message = $e->getMessage();
            $code    = 400;
            $status  = 'error';
            $url     = guard_url('help/help/' . $help->getRouteKey());
        }
        return compact('data', 'message', 'code', 'status', 'url');
    }

    /**
     * Remove the help.
     *
     * @param Model   $help
     *
     * @return Response
     */
    public function destroy(HelpRequest $request, Help $help)
    {
        try {
            $help->delete();
            $message = trans('messages.success.deleted', ['Module' => trans('help::help.name')]);
            $code    = 202;
            $status  = 'success';
            $url     = guard_url('help/help/0');
        } catch (Exception $e) {
            $message = $e->getMessage();
            $code    = 400;
            $status  = 'error';
            $url     = guard_url('help/help/' . $help->getRouteKey());
        }
        return compact('message', 'code', 'status', 'url');
    }

    /**
     * Return the form elements as json.
     *
     * @param String   $element
     *
     * @return json
     */
    public function form($element = 'fields')
    {
        $form = new Form();
        return $form->form($element, true);
    }

}
