<?php

namespace B2buy\Help\Http\Controllers;

use App\Http\Controllers\ResourceController as BaseController;
use Form;
use B2buy\Help\Http\Requests\HelpRequest;
use B2buy\Help\Interfaces\HelpRepositoryInterface;
use B2buy\Help\Models\Help;

/**
 * Resource controller class for help.
 */
class HelpResourceController extends BaseController
{

    /**
     * Initialize help resource controller.
     *
     * @param type HelpRepositoryInterface $help
     *
     * @return null
     */
    public function __construct(HelpRepositoryInterface $help)
    {
        parent::__construct();
        $this->repository = $help;
        $this->repository
            ->pushCriteria(\Litepie\Repository\Criteria\RequestCriteria::class)
            ->pushCriteria(\B2buy\Help\Repositories\Criteria\HelpResourceCriteria::class);
    }

    /**
     * Display a list of help.
     *
     * @return Response
     */
    public function index(HelpRequest $request)
    {
        $view = $this->response->theme->listView();

        if ($this->response->typeIs('json')) {
            $function = camel_case('get-' . $view);
            return $this->repository
                ->setPresenter(\B2buy\Help\Repositories\Presenter\HelpPresenter::class)
                ->$function();
        }

        $helps = $this->repository->paginate();

        return $this->response->setMetaTitle(trans('help::help.names'))
            ->view('help::help.index', true)
            ->data(compact('helps', 'view'))
            ->output();
    }

    /**
     * Display help.
     *
     * @param Request $request
     * @param Model   $help
     *
     * @return Response
     */
    public function show(HelpRequest $request, Help $help)
    {

        if ($help->exists) {
            $view = 'help::help.show';
        } else {
            $view = 'help::help.new';
        }

        return $this->response->setMetaTitle(trans('app.view') . ' ' . trans('help::help.name'))
            ->data(compact('help'))
            ->view($view, true)
            ->output();
    }

    /**
     * Show the form for creating a new help.
     *
     * @param Request $request
     *
     * @return Response
     */
    public function create(HelpRequest $request)
    {

        $help = $this->repository->newInstance([]);
        return $this->response->setMetaTitle(trans('app.new') . ' ' . trans('help::help.name')) 
            ->view('help::help.create', true) 
            ->data(compact('help'))
            ->output();
    }

    /**
     * Create new help.
     *
     * @param Request $request
     *
     * @return Response
     */
    public function store(HelpRequest $request)
    {
        try {
            $attributes              = $request->all();
            $attributes['user_id']   = user_id();
            $attributes['user_type'] = user_type();
            $help                 = $this->repository->create($attributes);

            return $this->response->message(trans('messages.success.created', ['Module' => trans('help::help.name')]))
                ->code(204)
                ->status('success')
                ->url(guard_url('help/help/' . $help->getRouteKey()))
                ->redirect();
        } catch (Exception $e) {
            return $this->response->message($e->getMessage())
                ->code(400)
                ->status('error')
                ->url(guard_url('/help/help'))
                ->redirect();
        }

    }

    /**
     * Show help for editing.
     *
     * @param Request $request
     * @param Model   $help
     *
     * @return Response
     */
    public function edit(HelpRequest $request, Help $help)
    {
        return $this->response->setMetaTitle(trans('app.edit') . ' ' . trans('help::help.name'))
            ->view('help::help.edit', true)
            ->data(compact('help'))
            ->output();
    }

    /**
     * Update the help.
     *
     * @param Request $request
     * @param Model   $help
     *
     * @return Response
     */
    public function update(HelpRequest $request, Help $help)
    {
        try {
            $attributes = $request->all();

            $help->update($attributes);
            return $this->response->message(trans('messages.success.updated', ['Module' => trans('help::help.name')]))
                ->code(204)
                ->status('success')
                ->url(guard_url('help/help/' . $help->getRouteKey()))
                ->redirect();
        } catch (Exception $e) {
            return $this->response->message($e->getMessage())
                ->code(400)
                ->status('error')
                ->url(guard_url('help/help/' . $help->getRouteKey()))
                ->redirect();
        }

    }

    /**
     * Remove the help.
     *
     * @param Model   $help
     *
     * @return Response
     */
    public function destroy(HelpRequest $request, Help $help)
    {
        try {

            $help->delete();
            return $this->response->message(trans('messages.success.deleted', ['Module' => trans('help::help.name')]))
                ->code(202)
                ->status('success')
                ->url(guard_url('help/help/0'))
                ->redirect();

        } catch (Exception $e) {

            return $this->response->message($e->getMessage())
                ->code(400)
                ->status('error')
                ->url(guard_url('help/help/' . $help->getRouteKey()))
                ->redirect();
        }

    }

    /**
     * Remove multiple help.
     *
     * @param Model   $help
     *
     * @return Response
     */
    public function delete(HelpRequest $request, $type)
    {
        try {
            $ids = hashids_decode($request->input('ids'));

            if ($type == 'purge') {
                $this->repository->purge($ids);
            } else {
                $this->repository->delete($ids);
            }

            return $this->response->message(trans('messages.success.deleted', ['Module' => trans('help::help.name')]))
                ->status("success")
                ->code(202)
                ->url(guard_url('help/help'))
                ->redirect();

        } catch (Exception $e) {

            return $this->response->message($e->getMessage())
                ->status("error")
                ->code(400)
                ->url(guard_url('/help/help'))
                ->redirect();
        }

    }

    /**
     * Restore deleted helps.
     *
     * @param Model   $help
     *
     * @return Response
     */
    public function restore(HelpRequest $request)
    {
        try {
            $ids = hashids_decode($request->input('ids'));
            $this->repository->restore($ids);

            return $this->response->message(trans('messages.success.restore', ['Module' => trans('help::help.name')]))
                ->status("success")
                ->code(202)
                ->url(guard_url('/help/help'))
                ->redirect();

        } catch (Exception $e) {

            return $this->response->message($e->getMessage())
                ->status("error")
                ->code(400)
                ->url(guard_url('/help/help/'))
                ->redirect();
        }

    }

}
