<?php

namespace Litecms\Blog\Http\Controllers;

use App\Http\Controllers\APIController as BaseController;
use Litecms\Blog\Http\Requests\BlogRequest;
use Litecms\Blog\Interfaces\BlogRepositoryInterface;
use Litecms\Blog\Models\Blog;
use Litecms\Blog\Forms\Blog as Form;

/**
 * APIController  class for blog.
 */
class BlogAPIController extends BaseController
{

    /**
     * Initialize blog resource controller.
     *
     * @param type BlogRepositoryInterface $blog
     *
     * @return null
     */
    public function __construct(BlogRepositoryInterface $blog)
    {
        parent::__construct();
        $this->repository = $blog;
        $this->repository
            ->pushCriteria(\Litepie\Repository\Criteria\RequestCriteria::class)
            ->pushCriteria(\Litecms\Blog\Repositories\Criteria\BlogResourceCriteria::class);
    }

    /**
     * Display a list of blog.
     *
     * @return Response
     */
    public function index(BlogRequest $request)
    {
        return $this->repository
            ->setPresenter(\Litecms\Blog\Repositories\Presenter\BlogPresenter::class)
            ->paginate();
    }

    /**
     * Display blog.
     *
     * @param Request $request
     * @param Model   $blog
     *
     * @return Response
     */
    public function show(BlogRequest $request, Blog $blog)
    {
        return $blog->setPresenter(\Litecms\Blog\Repositories\Presenter\BlogListPresenter::class);
        ;
    }

    /**
     * Create new blog.
     *
     * @param Request $request
     *
     * @return Response
     */
    public function store(BlogRequest $request)
    {
        try {
            $data              = $request->all();
            $data['user_id']   = user_id();
            $data['user_type'] = user_type();
            $data              = $this->repository->create($data);
            $message           = trans('messages.success.created', ['Module' => trans('blog::blog.name')]);
            $code              = 204;
            $status            = 'success';
            $url               = guard_url('blog/blog/' . $blog->getRouteKey());
        } catch (Exception $e) {
            $message = $e->getMessage();
            $code    = 400;
            $status  = 'error';
            $url     = guard_url('blog/blog');
        }
        return compact('data', 'message', 'code', 'status', 'url');
    }

    /**
     * Update the blog.
     *
     * @param Request $request
     * @param Model   $blog
     *
     * @return Response
     */
    public function update(BlogRequest $request, Blog $blog)
    {
        try {
            $data = $request->all();

            $blog->update($data);
            $message = trans('messages.success.updated', ['Module' => trans('blog::blog.name')]);
            $code    = 204;
            $status  = 'success';
            $url     = guard_url('blog/blog/' . $blog->getRouteKey());
        } catch (Exception $e) {
            $message = $e->getMessage();
            $code    = 400;
            $status  = 'error';
            $url     = guard_url('blog/blog/' . $blog->getRouteKey());
        }
        return compact('data', 'message', 'code', 'status', 'url');
    }

    /**
     * Remove the blog.
     *
     * @param Model   $blog
     *
     * @return Response
     */
    public function destroy(BlogRequest $request, Blog $blog)
    {
        try {
            $blog->delete();
            $message = trans('messages.success.deleted', ['Module' => trans('blog::blog.name')]);
            $code    = 202;
            $status  = 'success';
            $url     = guard_url('blog/blog/0');
        } catch (Exception $e) {
            $message = $e->getMessage();
            $code    = 400;
            $status  = 'error';
            $url     = guard_url('blog/blog/' . $blog->getRouteKey());
        }
        return compact('message', 'code', 'status', 'url');
    }

    /**
     * Return the form elements as json.
     *
     * @param String   $element
     *
     * @return json
     */
    public function form($element = 'fields')
    {
        $form = new Form();
        return $form->form($element, true);
    }

}
