<?php

namespace Superrobota\Language\Http\Controllers;

use App\Http\Controllers\ResourceController as BaseController;
use Form;
use Superrobota\Language\Http\Requests\LanguageRequest;
use Superrobota\Language\Interfaces\LanguageRepositoryInterface;
use Superrobota\Language\Models\Language;

/**
 * Resource controller class for language.
 */
class LanguageResourceController extends BaseController
{

    /**
     * Initialize language resource controller.
     *
     * @param type LanguageRepositoryInterface $language
     *
     * @return null
     */
    public function __construct(LanguageRepositoryInterface $language)
    {
        parent::__construct();
        $this->repository = $language;
        $this->repository
            ->pushCriteria(\Litepie\Repository\Criteria\RequestCriteria::class)
            ->pushCriteria(\Superrobota\Language\Repositories\Criteria\LanguageResourceCriteria::class);
    }

    /**
     * Display a list of language.
     *
     * @return Response
     */
    public function index(LanguageRequest $request)
    {
        $view = $this->response->theme->listView();

        if ($this->response->typeIs('json')) {
            $function = camel_case('get-' . $view);
            return $this->repository
                ->setPresenter(\Superrobota\Language\Repositories\Presenter\LanguagePresenter::class)
                ->$function();
        }

        $languages = $this->repository->paginate();

        return $this->response->title(trans('language::language.names'))
            ->view('language::language.index', true)
            ->data(compact('languages'))
            ->output();
    }

    /**
     * Display language.
     *
     * @param Request $request
     * @param Model   $language
     *
     * @return Response
     */
    public function show(LanguageRequest $request, Language $language)
    {

        if ($language->exists) {
            $view = 'language::language.show';
        } else {
            $view = 'language::language.new';
        }

        return $this->response->title(trans('app.view') . ' ' . trans('language::language.name'))
            ->data(compact('language'))
            ->view($view, true)
            ->output();
    }

    /**
     * Show the form for creating a new language.
     *
     * @param Request $request
     *
     * @return Response
     */
    public function create(LanguageRequest $request)
    {

        $language = $this->repository->newInstance([]);
        return $this->response->title(trans('app.new') . ' ' . trans('language::language.name')) 
            ->view('language::language.create', true) 
            ->data(compact('language'))
            ->output();
    }

    /**
     * Create new language.
     *
     * @param Request $request
     *
     * @return Response
     */
    public function store(LanguageRequest $request)
    {
        try {
            $attributes              = $request->all();
            $attributes['user_id']   = user_id();
            $attributes['user_type'] = user_type();
            $language                 = $this->repository->create($attributes);

            return $this->response->message(trans('messages.success.created', ['Module' => trans('language::language.name')]))
                ->code(204)
                ->status('success')
                ->url(guard_url('language/language/' . $language->getRouteKey()))
                ->redirect();
        } catch (Exception $e) {
            return $this->response->message($e->getMessage())
                ->code(400)
                ->status('error')
                ->url(guard_url('/language/language'))
                ->redirect();
        }

    }

    /**
     * Show language for editing.
     *
     * @param Request $request
     * @param Model   $language
     *
     * @return Response
     */
    public function edit(LanguageRequest $request, Language $language)
    {
        return $this->response->title(trans('app.edit') . ' ' . trans('language::language.name'))
            ->view('language::language.edit', true)
            ->data(compact('language'))
            ->output();
    }

    /**
     * Update the language.
     *
     * @param Request $request
     * @param Model   $language
     *
     * @return Response
     */
    public function update(LanguageRequest $request, Language $language)
    {
        try {
            $attributes = $request->all();

            $language->update($attributes);
            return $this->response->message(trans('messages.success.updated', ['Module' => trans('language::language.name')]))
                ->code(204)
                ->status('success')
                ->url(guard_url('language/language/' . $language->getRouteKey()))
                ->redirect();
        } catch (Exception $e) {
            return $this->response->message($e->getMessage())
                ->code(400)
                ->status('error')
                ->url(guard_url('language/language/' . $language->getRouteKey()))
                ->redirect();
        }

    }

    /**
     * Remove the language.
     *
     * @param Model   $language
     *
     * @return Response
     */
    public function destroy(LanguageRequest $request, Language $language)
    {
        try {

            $language->delete();
            return $this->response->message(trans('messages.success.deleted', ['Module' => trans('language::language.name')]))
                ->code(202)
                ->status('success')
                ->url(guard_url('language/language/0'))
                ->redirect();

        } catch (Exception $e) {

            return $this->response->message($e->getMessage())
                ->code(400)
                ->status('error')
                ->url(guard_url('language/language/' . $language->getRouteKey()))
                ->redirect();
        }

    }

    /**
     * Remove multiple language.
     *
     * @param Model   $language
     *
     * @return Response
     */
    public function delete(LanguageRequest $request, $type)
    {
        try {
            $ids = hashids_decode($request->input('ids'));

            if ($type == 'purge') {
                $this->repository->purge($ids);
            } else {
                $this->repository->delete($ids);
            }

            return $this->response->message(trans('messages.success.deleted', ['Module' => trans('language::language.name')]))
                ->status("success")
                ->code(202)
                ->url(guard_url('language/language'))
                ->redirect();

        } catch (Exception $e) {

            return $this->response->message($e->getMessage())
                ->status("error")
                ->code(400)
                ->url(guard_url('/language/language'))
                ->redirect();
        }

    }

    /**
     * Restore deleted languages.
     *
     * @param Model   $language
     *
     * @return Response
     */
    public function restore(LanguageRequest $request)
    {
        try {
            $ids = hashids_decode($request->input('ids'));
            $this->repository->restore($ids);

            return $this->response->message(trans('messages.success.restore', ['Module' => trans('language::language.name')]))
                ->status("success")
                ->code(202)
                ->url(guard_url('/language/language'))
                ->redirect();

        } catch (Exception $e) {

            return $this->response->message($e->getMessage())
                ->status("error")
                ->code(400)
                ->url(guard_url('/language/language/'))
                ->redirect();
        }

    }

}
