<?php

namespace Bixo\Message\Http\Controllers;

use Exception;
use Litepie\Http\Controllers\ResourceController as BaseController;
use Bixo\Message\Forms\Message as MessageForm;
use Bixo\Message\Http\Requests\MessageResourceRequest;
use Bixo\Message\Http\Resources\MessageResource;
use Bixo\Message\Http\Resources\MessagesCollection;
use Bixo\Message\Models\Message;
use Bixo\Message\Actions\MessageAction;
use Bixo\Message\Actions\MessageActions;


/**
 * Resource controller class for message.
 */
class MessageResourceController extends BaseController
{

    /**
     * Initialize message resource controller.
     *
     *
     * @return null
     */
    public function __construct()
    {
        parent::__construct();
        $this->middleware(function ($request, $next) {
            $this->form = MessageForm::grouped(false)
                            ->setAttributes()
                            ->toArray();
            $this->modules = $this->modules(config('bixo.message.modules'), 'message', guard_url('message'));
            return $next($request);
        });
    }

    /**
     * Display a list of message.
     *
     * @return Response
     */
    public function index(MessageResourceRequest $request)
    {
        $request = $request->all();
        $page = MessageActions::run('paginate', $request);

        $data = new MessagesCollection($page);

        $form = $this->form;
        $modules = $this->modules;

        return $this->response->setMetaTitle(trans('message::message.names'))
            ->view('message::message.index')
            ->data(compact('data', 'modules', 'form'))
            ->output();

    }

    /**
     * Display message.
     *
     * @param Request $request
     * @param Model   $message
     *
     * @return Response
     */
    public function show(MessageResourceRequest $request, Message $model)
    {
        $form = $this->form;
        $modules = $this->modules;
        $data = new MessageResource($model);
        return $this->response
            ->setMetaTitle(trans('app.view') . ' ' . trans('message::message.name'))
            ->data(compact('data', 'form', 'modules'))
            ->view('message::message.show')
            ->output();
    }

    /**
     * Show the form for creating a new message.
     *
     * @param Request $request
     *
     * @return Response
     */
    public function create(MessageResourceRequest $request, Message $model)
    {
        $form = $this->form;
        $modules = $this->modules;
        $data = new MessageResource($model);
        return $this->response->setMetaTitle(trans('app.new') . ' ' . trans('message::message.name'))
            ->view('message::message.create')
            ->data(compact('data', 'form', 'modules'))
            ->output();

    }

    /**
     * Create new message.
     *
     * @param Request $request
     *
     * @return Response
     */
    public function store(MessageResourceRequest $request, Message $model)
    {
        try {
            $request = $request->all();
            $model = MessageAction::run('store', $model, $request);
            $data = new MessageResource($model);
            return $this->response->message(trans('messages.success.created', ['Module' => trans('message::message.name')]))
                ->code(204)
                ->data(compact('data'))
                ->status('success')
                ->url(guard_url('message/message/' . $model->getRouteKey()))
                ->redirect();
        } catch (Exception $e) {
            return $this->response->message($e->getMessage())
                ->code(400)
                ->status('error')
                ->url(guard_url('/message/message'))
                ->redirect();
        }

    }

    /**
     * Show message for editing.
     *
     * @param Request $request
     * @param Model   $message
     *
     * @return Response
     */
    public function edit(MessageResourceRequest $request, Message $model)
    {
        $form = $this->form;
        $modules = $this->modules;
        $data = new MessageResource($model);
        // return view('message::message.edit', compact('data', 'form', 'modules'));

        return $this->response->setMetaTitle(trans('app.edit') . ' ' . trans('message::message.name'))
            ->view('message::message.edit')
            ->data(compact('data', 'form', 'modules'))
            ->output();

    }

    /**
     * Update the message.
     *
     * @param Request $request
     * @param Model   $message
     *
     * @return Response
     */
    public function update(MessageResourceRequest $request, Message $model)
    {
        try {
            $request = $request->all();
            $model = MessageAction::run('update', $model, $request);
            $data = new MessageResource($model);

            return $this->response->message(trans('messages.success.updated', ['Module' => trans('message::message.name')]))
                ->code(204)
                ->status('success')
                ->data(compact('data'))
                ->url(guard_url('message/message/' . $model->getRouteKey()))
                ->redirect();
        } catch (Exception $e) {
            return $this->response->message($e->getMessage())
                ->code(400)
                ->status('error')
                ->url(guard_url('message/message/' .  $model->getRouteKey()))
                ->redirect();
        }

    }

    /**
     * Remove the message.
     *
     * @param Model   $message
     *
     * @return Response
     */
    public function destroy(MessageResourceRequest $request, Message $model)
    {
        try {

            $request = $request->all();
            $model = MessageAction::run('destroy', $model, $request);
            $data = new MessageResource($model);

            return $this->response->message(trans('messages.success.deleted', ['Module' => trans('message::message.name')]))
                ->code(202)
                ->status('success')
                ->data(compact('data'))
                ->url(guard_url('message/message/0'))
                ->redirect();

        } catch (Exception $e) {

            return $this->response->message($e->getMessage())
                ->code(400)
                ->status('error')
                ->url(guard_url('message/message/' .  $model->getRouteKey()))
                ->redirect();
        }

    }
}
