<?php

namespace Bixo\Message\Http\Controllers;

use Exception;
use Litepie\Http\Controllers\ResourceController as BaseController;
use Bixo\Message\Forms\Recording as RecordingForm;
use Bixo\Message\Http\Requests\RecordingResourceRequest;
use Bixo\Message\Http\Resources\RecordingResource;
use Bixo\Message\Http\Resources\RecordingsCollection;
use Bixo\Message\Models\Recording;
use Bixo\Message\Actions\RecordingAction;
use Bixo\Message\Actions\RecordingActions;


/**
 * Resource controller class for recording.
 */
class RecordingResourceController extends BaseController
{

    /**
     * Initialize recording resource controller.
     *
     *
     * @return null
     */
    public function __construct()
    {
        parent::__construct();
        $this->middleware(function ($request, $next) {
            $this->form = RecordingForm::grouped(false)
                            ->setAttributes()
                            ->toArray();
            $this->modules = $this->modules(config('bixo.message.modules'), 'message', guard_url('message'));
            return $next($request);
        });
    }

    /**
     * Display a list of recording.
     *
     * @return Response
     */
    public function index(RecordingResourceRequest $request)
    {
        $request = $request->all();
        $page = RecordingActions::run('paginate', $request);

        $data = new RecordingsCollection($page);

        $form = $this->form;
        $modules = $this->modules;

        return $this->response->setMetaTitle(trans('message::recording.names'))
            ->view('message::recording.index')
            ->data(compact('data', 'modules', 'form'))
            ->output();

    }

    /**
     * Display recording.
     *
     * @param Request $request
     * @param Model   $recording
     *
     * @return Response
     */
    public function show(RecordingResourceRequest $request, Recording $model)
    {
        $form = $this->form;
        $modules = $this->modules;
        $data = new RecordingResource($model);
        return $this->response
            ->setMetaTitle(trans('app.view') . ' ' . trans('message::recording.name'))
            ->data(compact('data', 'form', 'modules'))
            ->view('message::recording.show')
            ->output();
    }

    /**
     * Show the form for creating a new recording.
     *
     * @param Request $request
     *
     * @return Response
     */
    public function create(RecordingResourceRequest $request, Recording $model)
    {
        $form = $this->form;
        $modules = $this->modules;
        $data = new RecordingResource($model);
        return $this->response->setMetaTitle(trans('app.new') . ' ' . trans('message::recording.name'))
            ->view('message::recording.create')
            ->data(compact('data', 'form', 'modules'))
            ->output();

    }

    /**
     * Create new recording.
     *
     * @param Request $request
     *
     * @return Response
     */
    public function store(RecordingResourceRequest $request, Recording $model)
    {
        try {
            $request = $request->all();
            $model = RecordingAction::run('store', $model, $request);
            $data = new RecordingResource($model);
            return $this->response->message(trans('messages.success.created', ['Module' => trans('message::recording.name')]))
                ->code(204)
                ->data(compact('data'))
                ->status('success')
                ->url(guard_url('message/recording/' . $model->getRouteKey()))
                ->redirect();
        } catch (Exception $e) {
            return $this->response->message($e->getMessage())
                ->code(400)
                ->status('error')
                ->url(guard_url('/message/recording'))
                ->redirect();
        }

    }

    /**
     * Show recording for editing.
     *
     * @param Request $request
     * @param Model   $recording
     *
     * @return Response
     */
    public function edit(RecordingResourceRequest $request, Recording $model)
    {
        $form = $this->form;
        $modules = $this->modules;
        $data = new RecordingResource($model);
        // return view('message::recording.edit', compact('data', 'form', 'modules'));

        return $this->response->setMetaTitle(trans('app.edit') . ' ' . trans('message::recording.name'))
            ->view('message::recording.edit')
            ->data(compact('data', 'form', 'modules'))
            ->output();

    }

    /**
     * Update the recording.
     *
     * @param Request $request
     * @param Model   $recording
     *
     * @return Response
     */
    public function update(RecordingResourceRequest $request, Recording $model)
    {
        try {
            $request = $request->all();
            $model = RecordingAction::run('update', $model, $request);
            $data = new RecordingResource($model);

            return $this->response->message(trans('messages.success.updated', ['Module' => trans('message::recording.name')]))
                ->code(204)
                ->status('success')
                ->data(compact('data'))
                ->url(guard_url('message/recording/' . $model->getRouteKey()))
                ->redirect();
        } catch (Exception $e) {
            return $this->response->message($e->getMessage())
                ->code(400)
                ->status('error')
                ->url(guard_url('message/recording/' .  $model->getRouteKey()))
                ->redirect();
        }

    }

    /**
     * Remove the recording.
     *
     * @param Model   $recording
     *
     * @return Response
     */
    public function destroy(RecordingResourceRequest $request, Recording $model)
    {
        try {

            $request = $request->all();
            $model = RecordingAction::run('destroy', $model, $request);
            $data = new RecordingResource($model);

            return $this->response->message(trans('messages.success.deleted', ['Module' => trans('message::recording.name')]))
                ->code(202)
                ->status('success')
                ->data(compact('data'))
                ->url(guard_url('message/recording/0'))
                ->redirect();

        } catch (Exception $e) {

            return $this->response->message($e->getMessage())
                ->code(400)
                ->status('error')
                ->url(guard_url('message/recording/' .  $model->getRouteKey()))
                ->redirect();
        }

    }
}
