<?php

namespace Bixo\Message\Listeners;

use Illuminate\Support\Facades\Notification;
use Illuminate\Support\Str;
use Bixo\Message\Events\RecordingWorkflow as RecordingWorkflowEvent;
use Bixo\Message\Notifications\RecordingWorkflow as RecordingWorkflowNotification;
use Litepie\Actions\Concerns\AsAction;

class RecordingWorkflow
{
    use AsAction;

    private $allowedTransitions = [
        'before' => [],
        'after' => ['publish', 'submit', 'approve'],
    ];

    public function handle(RecordingWorkflowEvent $event)
    {
        $function = Str::camel($event->transition);
        return $this->$function($event);
    }

    /**
     * Sends a notification to the client when the $recording is submitted.
     *
     * @param RecordingWorkflowEvent $event The event object.
     * @return void
     */
    public function submit(RecordingWorkflowEvent $event)
    {
        $client = $event->recording->client;
        Notification::send($client, new RecordingWorkflowNotification($event));
    }

    /**
     * Sends a notification to the client when the $recording is published.
     *
     * @param RecordingWorkflowEvent $event The event object.
     * @return void
     */
    public function publish(RecordingWorkflowEvent $event)
    {
        $client = $event->recording->client;
        Notification::send($client, new RecordingWorkflowNotification($event));
    }

    /**
     * Sends a notification to the client when the $recording is approved.
     *
     * @param RecordingWorkflowEvent $event The event object.
     * @return void
     */
    public function approve(RecordingWorkflowEvent $event)
    {
        $client = $event->recording->client;
        Notification::send($client, new RecordingWorkflowNotification($event));
    }

    /**
     * Handles the $recording workflow event as a listener.
     *
     * @param RecordingWorkflowEvent $event The event object.
     * @return void
     */
    public function asListener(RecordingWorkflowEvent $event)
    {
        if (($event->when == 'before' &&
            in_array($event->transition, $this->allowedTransitions['before']) ||
            $event->when == 'after' &&
            in_array($event->transition, $this->allowedTransitions['after']))
        ) {
            return $this->handle($event);
        }
    }
}
