<?php

namespace John\JohnPackage\Http\Controllers;

use App\Http\Controllers\ResourceController as BaseController;
use Form;
use John\JohnPackage\Http\Requests\ProjectRequest;
use John\JohnPackage\Interfaces\ProjectRepositoryInterface;
use John\JohnPackage\Models\Project;

/**
 * Resource controller class for project.
 */
class ProjectResourceController extends BaseController
{

    /**
     * Initialize project resource controller.
     *
     * @param type ProjectRepositoryInterface $project
     *
     * @return null
     */
    public function __construct(ProjectRepositoryInterface $project)
    {
        parent::__construct();
        $this->repository = $project;
        $this->repository
            ->pushCriteria(\Litepie\Repository\Criteria\RequestCriteria::class)
            ->pushCriteria(\John\JohnPackage\Repositories\Criteria\ProjectResourceCriteria::class);
    }

    /**
     * Display a list of project.
     *
     * @return Response
     */
    public function index(ProjectRequest $request)
    {
        $view = $this->response->theme->listView();

        if ($this->response->typeIs('json')) {
            $function = camel_case('get-' . $view);
            return $this->repository
                ->setPresenter(\John\JohnPackage\Repositories\Presenter\ProjectPresenter::class)
                ->$function();
        }

        $projects = $this->repository->paginate();

        return $this->response->title(trans('john_package::project.names'))
            ->view('john_package::project.index', true)
            ->data(compact('projects', 'view'))
            ->output();
    }

    /**
     * Display project.
     *
     * @param Request $request
     * @param Model   $project
     *
     * @return Response
     */
    public function show(ProjectRequest $request, Project $project)
    {

        if ($project->exists) {
            $view = 'john_package::project.show';
        } else {
            $view = 'john_package::project.new';
        }

        return $this->response->title(trans('app.view') . ' ' . trans('john_package::project.name'))
            ->data(compact('project'))
            ->view($view, true)
            ->output();
    }

    /**
     * Show the form for creating a new project.
     *
     * @param Request $request
     *
     * @return Response
     */
    public function create(ProjectRequest $request)
    {

        $project = $this->repository->newInstance([]);
        return $this->response->title(trans('app.new') . ' ' . trans('john_package::project.name')) 
            ->view('john_package::project.create', true) 
            ->data(compact('project'))
            ->output();
    }

    /**
     * Create new project.
     *
     * @param Request $request
     *
     * @return Response
     */
    public function store(ProjectRequest $request)
    {
        try {
            $attributes              = $request->all();
            $attributes['user_id']   = user_id();
            $attributes['user_type'] = user_type();
            $project                 = $this->repository->create($attributes);

            return $this->response->message(trans('messages.success.created', ['Module' => trans('john_package::project.name')]))
                ->code(204)
                ->status('success')
                ->url(guard_url('john_package/project/' . $project->getRouteKey()))
                ->redirect();
        } catch (Exception $e) {
            return $this->response->message($e->getMessage())
                ->code(400)
                ->status('error')
                ->url(guard_url('/john_package/project'))
                ->redirect();
        }

    }

    /**
     * Show project for editing.
     *
     * @param Request $request
     * @param Model   $project
     *
     * @return Response
     */
    public function edit(ProjectRequest $request, Project $project)
    {
        return $this->response->title(trans('app.edit') . ' ' . trans('john_package::project.name'))
            ->view('john_package::project.edit', true)
            ->data(compact('project'))
            ->output();
    }

    /**
     * Update the project.
     *
     * @param Request $request
     * @param Model   $project
     *
     * @return Response
     */
    public function update(ProjectRequest $request, Project $project)
    {
        try {
            $attributes = $request->all();

            $project->update($attributes);
            return $this->response->message(trans('messages.success.updated', ['Module' => trans('john_package::project.name')]))
                ->code(204)
                ->status('success')
                ->url(guard_url('john_package/project/' . $project->getRouteKey()))
                ->redirect();
        } catch (Exception $e) {
            return $this->response->message($e->getMessage())
                ->code(400)
                ->status('error')
                ->url(guard_url('john_package/project/' . $project->getRouteKey()))
                ->redirect();
        }

    }

    /**
     * Remove the project.
     *
     * @param Model   $project
     *
     * @return Response
     */
    public function destroy(ProjectRequest $request, Project $project)
    {
        try {

            $project->delete();
            return $this->response->message(trans('messages.success.deleted', ['Module' => trans('john_package::project.name')]))
                ->code(202)
                ->status('success')
                ->url(guard_url('john_package/project/0'))
                ->redirect();

        } catch (Exception $e) {

            return $this->response->message($e->getMessage())
                ->code(400)
                ->status('error')
                ->url(guard_url('john_package/project/' . $project->getRouteKey()))
                ->redirect();
        }

    }

    /**
     * Remove multiple project.
     *
     * @param Model   $project
     *
     * @return Response
     */
    public function delete(ProjectRequest $request, $type)
    {
        try {
            $ids = hashids_decode($request->input('ids'));

            if ($type == 'purge') {
                $this->repository->purge($ids);
            } else {
                $this->repository->delete($ids);
            }

            return $this->response->message(trans('messages.success.deleted', ['Module' => trans('john_package::project.name')]))
                ->status("success")
                ->code(202)
                ->url(guard_url('john_package/project'))
                ->redirect();

        } catch (Exception $e) {

            return $this->response->message($e->getMessage())
                ->status("error")
                ->code(400)
                ->url(guard_url('/john_package/project'))
                ->redirect();
        }

    }

    /**
     * Restore deleted projects.
     *
     * @param Model   $project
     *
     * @return Response
     */
    public function restore(ProjectRequest $request)
    {
        try {
            $ids = hashids_decode($request->input('ids'));
            $this->repository->restore($ids);

            return $this->response->message(trans('messages.success.restore', ['Module' => trans('john_package::project.name')]))
                ->status("success")
                ->code(202)
                ->url(guard_url('/john_package/project'))
                ->redirect();

        } catch (Exception $e) {

            return $this->response->message($e->getMessage())
                ->status("error")
                ->code(400)
                ->url(guard_url('/john_package/project/'))
                ->redirect();
        }

    }

}
