<?php

namespace Projects\Project\Http\Controllers;

use Exception;
use Litepie\Http\Controllers\ResourceController as BaseController;
use Litepie\Database\RequestScope;
use Projects\Project\Forms\Project as ProjectForm;
use Projects\Project\Http\Requests\ProjectRequest;
use Projects\Project\Http\Resources\ProjectResource;
use Projects\Project\Http\Resources\ProjectsCollection;
use Projects\Project\Models\Project;
use Projects\Project\Scopes\ProjectResourceScope;


/**
 * Resource controller class for project.
 */
class ProjectResourceController extends BaseController
{

    /**
     * Initialize project resource controller.
     *
     *
     * @return null
     */
    public function __construct()
    {
        parent::__construct();
        $this->form = ProjectForm::grouped(false)
                        ->setAttributes()
                        ->toArray();
        $this->modules = $this->modules(config('projects.project.modules'), 'project', guard_url('project'));
    }

    /**
     * Display a list of project.
     *
     * @return Response
     */
    public function index(ProjectRequest $request)
    {

        $pageLimit = $request->input('pageLimit', config('database.pagination.limit'));
        $page = Project::pushScope(new RequestScope())
            ->pushScope(new ProjectResourceScope())
            ->simplePaginate($pageLimit);

        $data = new ProjectsCollection($page);

        $form = $this->form;
        $modules = $this->modules;

        return $this->response->setMetaTitle(trans('project::project.names'))
            ->view('project::project.index')
            ->data(compact('data', 'modules', 'form'))
            ->output();

    }

    /**
     * Display project.
     *
     * @param Request $request
     * @param Model   $project
     *
     * @return Response
     */
    public function show(ProjectRequest $request, Project $model)
    {
        $form = $this->form;
        $modules = $this->modules;
        $data = new ProjectResource($model);

        return $this->response
            ->setMetaTitle(trans('app.view') . ' ' . trans('project::project.name'))
            ->data(compact('data', 'form', 'modules'))
            ->view('project::project.show')
            ->output();

    }

    /**
     * Show the form for creating a new project.
     *
     * @param Request $request
     *
     * @return Response
     */
    public function create(ProjectRequest $request, Project $model)
    {
        $form = $this->form;
        $modules = $this->modules;
        $data = new ProjectResource($model);
        return $this->response->setMetaTitle(trans('app.new') . ' ' . trans('project::project.name'))
            ->view('project::project.create')
            ->data(compact('data', 'form', 'modules'))
            ->output();

    }

    /**
     * Create new project.
     *
     * @param Request $request
     *
     * @return Response
     */
    public function store(ProjectRequest $request, Project $model)
    {
        try {
            $attributes = $request->all();
            $attributes['user_id'] = user_id();
            $attributes['user_type'] = user_type();
            $model = $model->create($attributes);
            $data = new ProjectResource($model);

            return $this->response->message(trans('messages.success.created', ['Module' => trans('project::project.name')]))
                ->code(204)
                ->data(compact('data'))
                ->status('success')
                ->url(guard_url('project/project/' . $model->getRouteKey()))
                ->redirect();
        } catch (Exception $e) {
            return $this->response->message($e->getMessage())
                ->code(400)
                ->status('error')
                ->url(guard_url('/project/project'))
                ->redirect();
        }

    }

    /**
     * Show project for editing.
     *
     * @param Request $request
     * @param Model   $project
     *
     * @return Response
     */
    public function edit(ProjectRequest $request, Project $model)
    {
        $form = $this->form;
        $modules = $this->modules;
        $data = new ProjectResource($model);

        return $this->response->setMetaTitle(trans('app.edit') . ' ' . trans('project::project.name'))
            ->view('project::project.edit')
            ->data(compact('data', 'form', 'modules'))
            ->output();

    }

    /**
     * Update the project.
     *
     * @param Request $request
     * @param Model   $project
     *
     * @return Response
     */
    public function update(ProjectRequest $request, Project $model)
    {
        try {
            $attributes = $request->all();
            $model = $model->update($attributes);
            $data = new ProjectResource($model);

            return $this->response->message(trans('messages.success.updated', ['Module' => trans('project::project.name')]))
                ->code(204)
                ->status('success')
                ->data(compact('data'))
                ->url(guard_url('project/project/' . $model->getRouteKey()))
                ->redirect();
        } catch (Exception $e) {
            return $this->response->message($e->getMessage())
                ->code(400)
                ->status('error')
                ->url(guard_url('project/project/' .  $model->getRouteKey()))
                ->redirect();
        }

    }

    /**
     * Remove the project.
     *
     * @param Model   $project
     *
     * @return Response
     */
    public function destroy(ProjectRequest $request, Project $model)
    {
        try {
            $model->delete();
            $data = new ProjectResource($model);

            return $this->response->message(trans('messages.success.deleted', ['Module' => trans('project::project.name')]))
                ->code(202)
                ->status('success')
                ->data(compact('data'))
                ->url(guard_url('project/project/0'))
                ->redirect();

        } catch (Exception $e) {

            return $this->response->message($e->getMessage())
                ->code(400)
                ->status('error')
                ->url(guard_url('project/project/' .  $model->getRouteKey()))
                ->redirect();
        }

    }
}
