<?php

namespace Litecms\RestaurantApp\Http\Controllers;

use Exception;
use Litepie\Http\Controllers\ResourceController as BaseController;
use Litepie\Database\RequestScope;
use Litecms\RestaurantApp\Forms\RestaurantApp as RestaurantAppForm;
use Litecms\RestaurantApp\Http\Requests\RestaurantAppRequest;
use Litecms\RestaurantApp\Http\Resources\RestaurantAppResource;
use Litecms\RestaurantApp\Http\Resources\RestaurantAppsCollection;
use Litecms\RestaurantApp\Models\RestaurantApp;
use Litecms\RestaurantApp\Scopes\RestaurantAppResourceScope;


/**
 * Resource controller class for restaurant_app.
 */
class RestaurantAppResourceController extends BaseController
{

    /**
     * Initialize restaurant_app resource controller.
     *
     *
     * @return null
     */
    public function __construct()
    {
        parent::__construct();
        $this->form = RestaurantAppForm::grouped(false)
                        ->setAttributes()
                        ->toArray();
        $this->modules = $this->modules(config('litecms.restaurant_app.modules'), 'restaurant_app', guard_url('restaurant_app'));
    }

    /**
     * Display a list of restaurant_app.
     *
     * @return Response
     */
    public function index(RestaurantAppRequest $request)
    {

        $pageLimit = $request->input('pageLimit', config('database.pagination.limit'));
        $page = RestaurantApp::pushScope(new RequestScope())
            ->pushScope(new RestaurantAppResourceScope())
            ->simplePaginate($pageLimit);

        $data = new RestaurantAppsCollection($page);

        $form = $this->form;
        $modules = $this->modules;

        return $this->response->setMetaTitle(trans('restaurant_app::restaurant_app.names'))
            ->view('restaurant_app::restaurant_app.index')
            ->data(compact('data', 'modules', 'form'))
            ->output();

    }

    /**
     * Display restaurant_app.
     *
     * @param Request $request
     * @param Model   $restaurant_app
     *
     * @return Response
     */
    public function show(RestaurantAppRequest $request, RestaurantApp $model)
    {
        $form = $this->form;
        $modules = $this->modules;
        $data = new RestaurantAppResource($model);

        return $this->response
            ->setMetaTitle(trans('app.view') . ' ' . trans('restaurant_app::restaurant_app.name'))
            ->data(compact('data', 'form', 'modules'))
            ->view('restaurant_app::restaurant_app.show')
            ->output();

    }

    /**
     * Show the form for creating a new restaurant_app.
     *
     * @param Request $request
     *
     * @return Response
     */
    public function create(RestaurantAppRequest $request, RestaurantApp $model)
    {
        $form = $this->form;
        $modules = $this->modules;
        $data = new RestaurantAppResource($model);
        return $this->response->setMetaTitle(trans('app.new') . ' ' . trans('restaurant_app::restaurant_app.name'))
            ->view('restaurant_app::restaurant_app.create')
            ->data(compact('data', 'form', 'modules'))
            ->output();

    }

    /**
     * Create new restaurant_app.
     *
     * @param Request $request
     *
     * @return Response
     */
    public function store(RestaurantAppRequest $request, RestaurantApp $model)
    {
        try {
            $attributes = $request->all();
            $attributes['user_id'] = user_id();
            $attributes['user_type'] = user_type();
            $model = $model->create($attributes);
            $data = new RestaurantAppResource($model);

            return $this->response->message(trans('messages.success.created', ['Module' => trans('restaurant_app::restaurant_app.name')]))
                ->code(204)
                ->data(compact('data'))
                ->status('success')
                ->url(guard_url('restaurant_app/restaurant_app/' . $model->getRouteKey()))
                ->redirect();
        } catch (Exception $e) {
            return $this->response->message($e->getMessage())
                ->code(400)
                ->status('error')
                ->url(guard_url('/restaurant_app/restaurant_app'))
                ->redirect();
        }

    }

    /**
     * Show restaurant_app for editing.
     *
     * @param Request $request
     * @param Model   $restaurant_app
     *
     * @return Response
     */
    public function edit(RestaurantAppRequest $request, RestaurantApp $model)
    {
        $form = $this->form;
        $modules = $this->modules;
        $data = new RestaurantAppResource($model);

        return $this->response->setMetaTitle(trans('app.edit') . ' ' . trans('restaurant_app::restaurant_app.name'))
            ->view('restaurant_app::restaurant_app.edit')
            ->data(compact('data', 'form', 'modules'))
            ->output();

    }

    /**
     * Update the restaurant_app.
     *
     * @param Request $request
     * @param Model   $restaurant_app
     *
     * @return Response
     */
    public function update(RestaurantAppRequest $request, RestaurantApp $model)
    {
        try {
            $attributes = $request->all();
            $model = $model->update($attributes);
            $data = new RestaurantAppResource($model);

            return $this->response->message(trans('messages.success.updated', ['Module' => trans('restaurant_app::restaurant_app.name')]))
                ->code(204)
                ->status('success')
                ->data(compact('data'))
                ->url(guard_url('restaurant_app/restaurant_app/' . $model->getRouteKey()))
                ->redirect();
        } catch (Exception $e) {
            return $this->response->message($e->getMessage())
                ->code(400)
                ->status('error')
                ->url(guard_url('restaurant_app/restaurant_app/' .  $model->getRouteKey()))
                ->redirect();
        }

    }

    /**
     * Remove the restaurant_app.
     *
     * @param Model   $restaurant_app
     *
     * @return Response
     */
    public function destroy(RestaurantAppRequest $request, RestaurantApp $model)
    {
        try {
            $model->delete();
            $data = new RestaurantAppResource($model);

            return $this->response->message(trans('messages.success.deleted', ['Module' => trans('restaurant_app::restaurant_app.name')]))
                ->code(202)
                ->status('success')
                ->data(compact('data'))
                ->url(guard_url('restaurant_app/restaurant_app/0'))
                ->redirect();

        } catch (Exception $e) {

            return $this->response->message($e->getMessage())
                ->code(400)
                ->status('error')
                ->url(guard_url('restaurant_app/restaurant_app/' .  $model->getRouteKey()))
                ->redirect();
        }

    }
}
