<?php

namespace Bixo\Source\Http\Controllers;

use Exception;
use Litepie\Http\Controllers\ResourceController as BaseController;
use Bixo\Source\Forms\Source as SourceForm;
use Bixo\Source\Http\Requests\SourceResourceRequest;
use Bixo\Source\Http\Resources\SourceResource;
use Bixo\Source\Http\Resources\SourcesCollection;
use Bixo\Source\Models\Source;
use Bixo\Source\Actions\SourceAction;
use Bixo\Source\Actions\SourceActions;


/**
 * Resource controller class for source.
 */
class SourceResourceController extends BaseController
{

    /**
     * Initialize source resource controller.
     *
     *
     * @return null
     */
    public function __construct()
    {
        parent::__construct();
        $this->middleware(function ($request, $next) {
            $this->form = SourceForm::grouped(false)
                            ->setAttributes()
                            ->toArray();
            $this->modules = $this->modules(config('bixo.source.modules'), 'source', guard_url('source'));
            return $next($request);
        });
    }

    /**
     * Display a list of source.
     *
     * @return Response
     */
    public function index(SourceResourceRequest $request)
    {
        $request = $request->all();
        $page = SourceActions::run('paginate', $request);

        $data = new SourcesCollection($page);

        $form = $this->form;
        $modules = $this->modules;

        return $this->response->setMetaTitle(trans('source::source.names'))
            ->view('source::source.index')
            ->data(compact('data', 'modules', 'form'))
            ->output();

    }

    /**
     * Display source.
     *
     * @param Request $request
     * @param Model   $source
     *
     * @return Response
     */
    public function show(SourceResourceRequest $request, Source $model)
    {
        $form = $this->form;
        $modules = $this->modules;
        $data = new SourceResource($model);
        return $this->response
            ->setMetaTitle(trans('app.view') . ' ' . trans('source::source.name'))
            ->data(compact('data', 'form', 'modules'))
            ->view('source::source.show')
            ->output();
    }

    /**
     * Show the form for creating a new source.
     *
     * @param Request $request
     *
     * @return Response
     */
    public function create(SourceResourceRequest $request, Source $model)
    {
        $form = $this->form;
        $modules = $this->modules;
        $data = new SourceResource($model);
        return $this->response->setMetaTitle(trans('app.new') . ' ' . trans('source::source.name'))
            ->view('source::source.create')
            ->data(compact('data', 'form', 'modules'))
            ->output();

    }

    /**
     * Create new source.
     *
     * @param Request $request
     *
     * @return Response
     */
    public function store(SourceResourceRequest $request, Source $model)
    {
        try {
            $request = $request->all();
            $model = SourceAction::run('store', $model, $request);
            $data = new SourceResource($model);
            return $this->response->message(trans('messages.success.created', ['Module' => trans('source::source.name')]))
                ->code(204)
                ->data(compact('data'))
                ->status('success')
                ->url(guard_url('source/source/' . $model->getRouteKey()))
                ->redirect();
        } catch (Exception $e) {
            return $this->response->message($e->getMessage())
                ->code(400)
                ->status('error')
                ->url(guard_url('/source/source'))
                ->redirect();
        }

    }

    /**
     * Show source for editing.
     *
     * @param Request $request
     * @param Model   $source
     *
     * @return Response
     */
    public function edit(SourceResourceRequest $request, Source $model)
    {
        $form = $this->form;
        $modules = $this->modules;
        $data = new SourceResource($model);
        // return view('source::source.edit', compact('data', 'form', 'modules'));

        return $this->response->setMetaTitle(trans('app.edit') . ' ' . trans('source::source.name'))
            ->view('source::source.edit')
            ->data(compact('data', 'form', 'modules'))
            ->output();

    }

    /**
     * Update the source.
     *
     * @param Request $request
     * @param Model   $source
     *
     * @return Response
     */
    public function update(SourceResourceRequest $request, Source $model)
    {
        try {
            $request = $request->all();
            $model = SourceAction::run('update', $model, $request);
            $data = new SourceResource($model);

            return $this->response->message(trans('messages.success.updated', ['Module' => trans('source::source.name')]))
                ->code(204)
                ->status('success')
                ->data(compact('data'))
                ->url(guard_url('source/source/' . $model->getRouteKey()))
                ->redirect();
        } catch (Exception $e) {
            return $this->response->message($e->getMessage())
                ->code(400)
                ->status('error')
                ->url(guard_url('source/source/' .  $model->getRouteKey()))
                ->redirect();
        }

    }

    /**
     * Remove the source.
     *
     * @param Model   $source
     *
     * @return Response
     */
    public function destroy(SourceResourceRequest $request, Source $model)
    {
        try {

            $request = $request->all();
            $model = SourceAction::run('destroy', $model, $request);
            $data = new SourceResource($model);

            return $this->response->message(trans('messages.success.deleted', ['Module' => trans('source::source.name')]))
                ->code(202)
                ->status('success')
                ->data(compact('data'))
                ->url(guard_url('source/source/0'))
                ->redirect();

        } catch (Exception $e) {

            return $this->response->message($e->getMessage())
                ->code(400)
                ->status('error')
                ->url(guard_url('source/source/' .  $model->getRouteKey()))
                ->redirect();
        }

    }
}
