<?php

namespace Bixo\Source\Listeners;

use Illuminate\Support\Facades\Notification;
use Illuminate\Support\Str;
use Bixo\Source\Events\SourceWorkflow as SourceWorkflowEvent;
use Bixo\Source\Notifications\SourceWorkflow as SourceWorkflowNotification;
use Litepie\Actions\Concerns\AsAction;

class SourceWorkflow
{
    use AsAction;

    private $allowedTransitions = [
        'before' => [],
        'after' => ['publish', 'submit', 'approve'],
    ];

    public function handle(SourceWorkflowEvent $event)
    {
        $function = Str::camel($event->transition);
        return $this->$function($event);
    }

    /**
     * Sends a notification to the client when the $source is submitted.
     *
     * @param SourceWorkflowEvent $event The event object.
     * @return void
     */
    public function submit(SourceWorkflowEvent $event)
    {
        $client = $event->source->client;
        Notification::send($client, new SourceWorkflowNotification($event));
    }

    /**
     * Sends a notification to the client when the $source is published.
     *
     * @param SourceWorkflowEvent $event The event object.
     * @return void
     */
    public function publish(SourceWorkflowEvent $event)
    {
        $client = $event->source->client;
        Notification::send($client, new SourceWorkflowNotification($event));
    }

    /**
     * Sends a notification to the client when the $source is approved.
     *
     * @param SourceWorkflowEvent $event The event object.
     * @return void
     */
    public function approve(SourceWorkflowEvent $event)
    {
        $client = $event->source->client;
        Notification::send($client, new SourceWorkflowNotification($event));
    }

    /**
     * Handles the $source workflow event as a listener.
     *
     * @param SourceWorkflowEvent $event The event object.
     * @return void
     */
    public function asListener(SourceWorkflowEvent $event)
    {
        if (($event->when == 'before' &&
            in_array($event->transition, $this->allowedTransitions['before']) ||
            $event->when == 'after' &&
            in_array($event->transition, $this->allowedTransitions['after']))
        ) {
            return $this->handle($event);
        }
    }
}
