<?php

namespace Liteerp\Source\Http\Controllers;

use App\Http\Controllers\ResourceController as BaseController;
use Form;
use Liteerp\Source\Http\Requests\SourceRequest;
use Liteerp\Source\Interfaces\SourceRepositoryInterface;
use Liteerp\Source\Models\Source;

/**
 * Resource controller class for source.
 */
class SourceResourceController extends BaseController
{

    /**
     * Initialize source resource controller.
     *
     * @param type SourceRepositoryInterface $source
     *
     * @return null
     */
    public function __construct(SourceRepositoryInterface $source)
    {
        parent::__construct();
        $this->repository = $source;
        $this->repository
            ->pushCriteria(\Litepie\Repository\Criteria\RequestCriteria::class)
            ->pushCriteria(\Liteerp\Source\Repositories\Criteria\SourceResourceCriteria::class);
    }

    /**
     * Display a list of source.
     *
     * @return Response
     */
    public function index(SourceRequest $request)
    {
        $view = $this->response->theme->listView();

        if ($this->response->typeIs('json')) {
            $function = camel_case('get-' . $view);
            return $this->repository
                ->setPresenter(\Liteerp\Source\Repositories\Presenter\SourcePresenter::class)
                ->$function();
        }

        $sources = $this->repository->paginate();

        return $this->response->title(trans('source::source.names'))
            ->view('source::source.index', true)
            ->data(compact('sources'))
            ->output();
    }

    /**
     * Display source.
     *
     * @param Request $request
     * @param Model   $source
     *
     * @return Response
     */
    public function show(SourceRequest $request, Source $source)
    {

        if ($source->exists) {
            $view = 'source::source.show';
        } else {
            $view = 'source::source.new';
        }

        return $this->response->title(trans('app.view') . ' ' . trans('source::source.name'))
            ->data(compact('source'))
            ->view($view, true)
            ->output();
    }

    /**
     * Show the form for creating a new source.
     *
     * @param Request $request
     *
     * @return Response
     */
    public function create(SourceRequest $request)
    {

        $source = $this->repository->newInstance([]);
        return $this->response->title(trans('app.new') . ' ' . trans('source::source.name')) 
            ->view('source::source.create', true) 
            ->data(compact('source'))
            ->output();
    }

    /**
     * Create new source.
     *
     * @param Request $request
     *
     * @return Response
     */
    public function store(SourceRequest $request)
    {
        try {
            $attributes              = $request->all();
            $attributes['user_id']   = user_id();
            $attributes['user_type'] = user_type();
            $source                 = $this->repository->create($attributes);

            return $this->response->message(trans('messages.success.created', ['Module' => trans('source::source.name')]))
                ->code(204)
                ->status('success')
                ->url(guard_url('source/source/' . $source->getRouteKey()))
                ->redirect();
        } catch (Exception $e) {
            return $this->response->message($e->getMessage())
                ->code(400)
                ->status('error')
                ->url(guard_url('/source/source'))
                ->redirect();
        }

    }

    /**
     * Show source for editing.
     *
     * @param Request $request
     * @param Model   $source
     *
     * @return Response
     */
    public function edit(SourceRequest $request, Source $source)
    {
        return $this->response->title(trans('app.edit') . ' ' . trans('source::source.name'))
            ->view('source::source.edit', true)
            ->data(compact('source'))
            ->output();
    }

    /**
     * Update the source.
     *
     * @param Request $request
     * @param Model   $source
     *
     * @return Response
     */
    public function update(SourceRequest $request, Source $source)
    {
        try {
            $attributes = $request->all();

            $source->update($attributes);
            return $this->response->message(trans('messages.success.updated', ['Module' => trans('source::source.name')]))
                ->code(204)
                ->status('success')
                ->url(guard_url('source/source/' . $source->getRouteKey()))
                ->redirect();
        } catch (Exception $e) {
            return $this->response->message($e->getMessage())
                ->code(400)
                ->status('error')
                ->url(guard_url('source/source/' . $source->getRouteKey()))
                ->redirect();
        }

    }

    /**
     * Remove the source.
     *
     * @param Model   $source
     *
     * @return Response
     */
    public function destroy(SourceRequest $request, Source $source)
    {
        try {

            $source->delete();
            return $this->response->message(trans('messages.success.deleted', ['Module' => trans('source::source.name')]))
                ->code(202)
                ->status('success')
                ->url(guard_url('source/source/0'))
                ->redirect();

        } catch (Exception $e) {

            return $this->response->message($e->getMessage())
                ->code(400)
                ->status('error')
                ->url(guard_url('source/source/' . $source->getRouteKey()))
                ->redirect();
        }

    }

    /**
     * Remove multiple source.
     *
     * @param Model   $source
     *
     * @return Response
     */
    public function delete(SourceRequest $request, $type)
    {
        try {
            $ids = hashids_decode($request->input('ids'));

            if ($type == 'purge') {
                $this->repository->purge($ids);
            } else {
                $this->repository->delete($ids);
            }

            return $this->response->message(trans('messages.success.deleted', ['Module' => trans('source::source.name')]))
                ->status("success")
                ->code(202)
                ->url(guard_url('source/source'))
                ->redirect();

        } catch (Exception $e) {

            return $this->response->message($e->getMessage())
                ->status("error")
                ->code(400)
                ->url(guard_url('/source/source'))
                ->redirect();
        }

    }

    /**
     * Restore deleted sources.
     *
     * @param Model   $source
     *
     * @return Response
     */
    public function restore(SourceRequest $request)
    {
        try {
            $ids = hashids_decode($request->input('ids'));
            $this->repository->restore($ids);

            return $this->response->message(trans('messages.success.restore', ['Module' => trans('source::source.name')]))
                ->status("success")
                ->code(202)
                ->url(guard_url('/source/source'))
                ->redirect();

        } catch (Exception $e) {

            return $this->response->message($e->getMessage())
                ->status("error")
                ->code(400)
                ->url(guard_url('/source/source/'))
                ->redirect();
        }

    }

}
