<?php

namespace Lavalite\Tasklist\Http\Controllers;

use App\Http\Controllers\AdminController as AdminController;
use Form;
use Lavalite\Tasklist\Http\Requests\TaskAdminRequest;
use Lavalite\Tasklist\Interfaces\TaskRepositoryInterface;
use Lavalite\Tasklist\Models\Task;

/**
 * Admin controller class.
 */
class TaskAdminController extends AdminController
{
    /**
     * Initialize task controller.
     *
     * @param type TaskRepositoryInterface $task
     *
     * @return type
     */
    public function __construct(TaskRepositoryInterface $task)
    {
        parent::__construct();
        $this->repository = $task;
    }

    /**
     * Display a list of task.
     *
     * @return Response
     */
    public function index(TaskAdminRequest $request)
    {
        $tasks  = $this->repository->setPresenter('\\Lavalite\\Tasklist\\Repositories\\Presenter\\TaskListPresenter')
                                            ->scopeQuery(function($query){
                                                return $query->orderBy('id','DESC');
                                            })->paginate();
        $this   ->theme->prependTitle(trans('tasklist::task.names').' :: ');
        $view   = $this->theme->of('tasklist::admin.task.index')->render();

        $this->responseCode = 200;
        $this->responseMessage = trans('messages.success.loaded', ['Module' => trans('tasklist::task.name')]);
        $this->responseData = $tasks['data'];
        $this->responseMeta = $tasks['meta'];
        $this->responseView = $view;
        $this->responseRedirect = '';
        return $this->respond($request);
    }

    /**
     * Display task.
     *
     * @param Request $request
     * @param int     $id
     *
     * @return Response
     */
    public function show(TaskAdminRequest $request, Task $task)
    {
        if (!$task->exists) {
            $this->responseCode = 404;
            $this->responseMessage = trans('messages.success.notfound', ['Module' => trans('tasklist::task.name')]);
            $this->responseView = view('tasklist::admin.task.new');
            return $this -> respond($request);
        }

        Form::populate($task);
        $this->responseCode = 200;
        $this->responseMessage = trans('messages.success.loaded', ['Module' => trans('tasklist::task.name')]);
        $this->responseData = $task;
        $this->responseView = view('tasklist::admin.task.show', compact('task'));
        return $this -> respond($request);
    }

    /**
     * Show the form for creating a new task.
     *
     * @param Request $request
     *
     * @return Response
     */
    public function create(TaskAdminRequest $request)
    {

        $task = $this->repository->newInstance([]);

        Form::populate($task);

        $this->responseCode = 200;
        $this->responseMessage = trans('messages.success.loaded', ['Module' => trans('tasklist::task.name')]);
        $this->responseData = $task;
        $this->responseView = view('tasklist::admin.task.create', compact('task'));
        return $this -> respond($request);

    }

    /**
     * Create new task.
     *
     * @param Request $request
     *
     * @return Response
     */
    public function store(TaskAdminRequest $request)
    {
        try {
            $attributes = $request->all();
            $attributes['user_id'] = user_id();
            $task = $this->repository->create($attributes);

            $this->responseCode = 201;
            $this->responseMessage = trans('messages.success.created', ['Module' => trans('tasklist::task.name')]);
            $this->responseData = $task;
            $this->responseMeta = '';
            $this->responseRedirect = trans_url('/admin/tasklist/task/'.$task->getRouteKey());
            $this->responseView = view('tasklist::admin.task.create', compact('task'));

            return $this -> respond($request);

        } catch (Exception $e) {
            $this->responseCode = 400;
            $this->responseMessage = $e->getMessage();
            return $this -> respond($request);
        }
    }

    /**
     * Show task for editing.
     *
     * @param Request $request
     * @param int     $id
     *
     * @return Response
     */
    public function edit(TaskAdminRequest $request, Task $task)
    {
        Form::populate($task);
        $this->responseCode = 200;
        $this->responseMessage = trans('messages.success.loaded', ['Module' => trans('tasklist::task.name')]);
        $this->responseData = $task;
        $this->responseView = view('tasklist::admin.task.edit', compact('task'));

        return $this -> respond($request);
    }

    /**
     * Update the task.
     *
     * @param Request $request
     * @param int     $id
     *
     * @return Response
     */
    public function update(TaskAdminRequest $request, Task $task)
    {
        try {

            $attributes = $request->all();

            $task->update($attributes);

            $this->responseCode = 204;
            $this->responseMessage = trans('messages.success.updated', ['Module' => trans('tasklist::task.name')]);
            $this->responseData = $task;
            $this->responseRedirect = trans_url('/admin/tasklist/task/'.$task->getRouteKey());

            return $this -> respond($request);

        } catch (Exception $e) {

            $this->responseCode = 400;
            $this->responseMessage = $e->getMessage();
            $this->responseRedirect = trans_url('/admin/tasklist/task/'.$task->getRouteKey());

            return $this -> respond($request);
        }
    }

    /**
     * Remove the task.
     *
     * @param int $id
     *
     * @return Response
     */
    public function destroy(TaskAdminRequest $request, Task $task)
    {

        try {

            $t = $task->delete();

            $this->responseCode = 202;
            $this->responseMessage = trans('messages.success.deleted', ['Module' => trans('tasklist::task.name')]);
            $this->responseRedirect = trans_url('/admin/tasklist/task/0');

            return $this -> respond($request);

        } catch (Exception $e) {

            $this->responseCode = 400;
            $this->responseMessage = $e->getMessage();
            $this->responseRedirect = trans_url('/admin/tasklist/task/'.$task->getRouteKey());

            return $this -> respond($request);

        }
    }
}
