<?php

namespace Testimonials\Testimonial\Http\Controllers;

use Exception;
use Litepie\Http\Controllers\ResourceController as BaseController;
use Litepie\Database\RequestScope;
use Testimonials\Testimonial\Forms\Testimonials as TestimonialsForm;
use Testimonials\Testimonial\Http\Requests\TestimonialsRequest;
use Testimonials\Testimonial\Http\Resources\TestimonialsResource;
use Testimonials\Testimonial\Http\Resources\TestimonialsCollection;
use Testimonials\Testimonial\Models\Testimonials;
use Testimonials\Testimonial\Scopes\TestimonialsResourceScope;


/**
 * Resource controller class for testimonials.
 */
class TestimonialsResourceController extends BaseController
{

    /**
     * Initialize testimonials resource controller.
     *
     *
     * @return null
     */
    public function __construct()
    {
        parent::__construct();
        $this->form = TestimonialsForm::grouped(false)
                        ->setAttributes()
                        ->toArray();
        $this->modules = $this->modules(config('testimonials.testimonial.modules'), 'testimonial', guard_url('testimonial'));
    }

    /**
     * Display a list of testimonials.
     *
     * @return Response
     */
    public function index(TestimonialsRequest $request)
    {

        $pageLimit = $request->input('pageLimit', config('database.pagination.limit'));
        $page = Testimonials::pushScope(new RequestScope())
            ->pushScope(new TestimonialsResourceScope())
            ->simplePaginate($pageLimit);

        $data = new TestimonialsCollection($page);

        $form = $this->form;
        $modules = $this->modules;

        return $this->response->setMetaTitle(trans('testimonial::testimonials.names'))
            ->view('testimonial::testimonials.index')
            ->data(compact('data', 'modules', 'form'))
            ->output();

    }

    /**
     * Display testimonials.
     *
     * @param Request $request
     * @param Model   $testimonials
     *
     * @return Response
     */
    public function show(TestimonialsRequest $request, Testimonials $model)
    {
        $form = $this->form;
        $modules = $this->modules;
        $data = new TestimonialsResource($model);

        return $this->response
            ->setMetaTitle(trans('app.view') . ' ' . trans('testimonial::testimonials.name'))
            ->data(compact('data', 'form', 'modules'))
            ->view('testimonial::testimonials.show')
            ->output();

    }

    /**
     * Show the form for creating a new testimonials.
     *
     * @param Request $request
     *
     * @return Response
     */
    public function create(TestimonialsRequest $request, Testimonials $model)
    {
        $form = $this->form;
        $modules = $this->modules;
        $data = new TestimonialsResource($model);
        return $this->response->setMetaTitle(trans('app.new') . ' ' . trans('testimonial::testimonials.name'))
            ->view('testimonial::testimonials.create')
            ->data(compact('data', 'form', 'modules'))
            ->output();

    }

    /**
     * Create new testimonials.
     *
     * @param Request $request
     *
     * @return Response
     */
    public function store(TestimonialsRequest $request, Testimonials $model)
    {
        try {
            $attributes = $request->all();
            $attributes['user_id'] = user_id();
            $attributes['user_type'] = user_type();
            $model = $model->create($attributes);
            $data = new TestimonialsResource($model);

            return $this->response->message(trans('messages.success.created', ['Module' => trans('testimonial::testimonials.name')]))
                ->code(204)
                ->data(compact('data'))
                ->status('success')
                ->url(guard_url('testimonial/testimonials/' . $model->getRouteKey()))
                ->redirect();
        } catch (Exception $e) {
            return $this->response->message($e->getMessage())
                ->code(400)
                ->status('error')
                ->url(guard_url('/testimonial/testimonials'))
                ->redirect();
        }

    }

    /**
     * Show testimonials for editing.
     *
     * @param Request $request
     * @param Model   $testimonials
     *
     * @return Response
     */
    public function edit(TestimonialsRequest $request, Testimonials $model)
    {
        $form = $this->form;
        $modules = $this->modules;
        $data = new TestimonialsResource($model);

        return $this->response->setMetaTitle(trans('app.edit') . ' ' . trans('testimonial::testimonials.name'))
            ->view('testimonial::testimonials.edit')
            ->data(compact('data', 'form', 'modules'))
            ->output();

    }

    /**
     * Update the testimonials.
     *
     * @param Request $request
     * @param Model   $testimonials
     *
     * @return Response
     */
    public function update(TestimonialsRequest $request, Testimonials $model)
    {
        try {
            $attributes = $request->all();
            $model = $model->update($attributes);
            $data = new TestimonialsResource($model);

            return $this->response->message(trans('messages.success.updated', ['Module' => trans('testimonial::testimonials.name')]))
                ->code(204)
                ->status('success')
                ->data(compact('data'))
                ->url(guard_url('testimonial/testimonials/' . $model->getRouteKey()))
                ->redirect();
        } catch (Exception $e) {
            return $this->response->message($e->getMessage())
                ->code(400)
                ->status('error')
                ->url(guard_url('testimonial/testimonials/' .  $model->getRouteKey()))
                ->redirect();
        }

    }

    /**
     * Remove the testimonials.
     *
     * @param Model   $testimonials
     *
     * @return Response
     */
    public function destroy(TestimonialsRequest $request, Testimonials $model)
    {
        try {
            $model->delete();
            $data = new TestimonialsResource($model);

            return $this->response->message(trans('messages.success.deleted', ['Module' => trans('testimonial::testimonials.name')]))
                ->code(202)
                ->status('success')
                ->data(compact('data'))
                ->url(guard_url('testimonial/testimonials/0'))
                ->redirect();

        } catch (Exception $e) {

            return $this->response->message($e->getMessage())
                ->code(400)
                ->status('error')
                ->url(guard_url('testimonial/testimonials/' .  $model->getRouteKey()))
                ->redirect();
        }

    }
}
