<?php

namespace Testing\Testing\Http\Controllers;

use App\Http\Controllers\ResourceController as BaseController;
use Form;
use Testing\Testing\Http\Requests\TestingRequest;
use Testing\Testing\Interfaces\TestingRepositoryInterface;
use Testing\Testing\Models\Testing;

/**
 * Resource controller class for testing.
 */
class TestingResourceController extends BaseController
{

    /**
     * Initialize testing resource controller.
     *
     * @param type TestingRepositoryInterface $testing
     *
     * @return null
     */
    public function __construct(TestingRepositoryInterface $testing)
    {
        parent::__construct();
        $this->repository = $testing;
        $this->repository
            ->pushCriteria(\Litepie\Repository\Criteria\RequestCriteria::class)
            ->pushCriteria(\Testing\Testing\Repositories\Criteria\TestingResourceCriteria::class);
    }

    /**
     * Display a list of testing.
     *
     * @return Response
     */
    public function index(TestingRequest $request)
    {
        $view = $this->response->theme->listView();

        if ($this->response->typeIs('json')) {
            $function = camel_case('get-' . $view);
            return $this->repository
                ->setPresenter(\Testing\Testing\Repositories\Presenter\TestingPresenter::class)
                ->$function();
        }

        $testings = $this->repository->paginate();

        return $this->response->title(trans('testing::testing.names'))
            ->view('testing::testing.index', true)
            ->data(compact('testings'))
            ->output();
    }

    /**
     * Display testing.
     *
     * @param Request $request
     * @param Model   $testing
     *
     * @return Response
     */
    public function show(TestingRequest $request, Testing $testing)
    {

        if ($testing->exists) {
            $view = 'testing::testing.show';
        } else {
            $view = 'testing::testing.new';
        }

        return $this->response->title(trans('app.view') . ' ' . trans('testing::testing.name'))
            ->data(compact('testing'))
            ->view($view, true)
            ->output();
    }

    /**
     * Show the form for creating a new testing.
     *
     * @param Request $request
     *
     * @return Response
     */
    public function create(TestingRequest $request)
    {

        $testing = $this->repository->newInstance([]);
        return $this->response->title(trans('app.new') . ' ' . trans('testing::testing.name')) 
            ->view('testing::testing.create', true) 
            ->data(compact('testing'))
            ->output();
    }

    /**
     * Create new testing.
     *
     * @param Request $request
     *
     * @return Response
     */
    public function store(TestingRequest $request)
    {
        try {
            $attributes              = $request->all();
            $attributes['user_id']   = user_id();
            $attributes['user_type'] = user_type();
            $testing                 = $this->repository->create($attributes);

            return $this->response->message(trans('messages.success.created', ['Module' => trans('testing::testing.name')]))
                ->code(204)
                ->status('success')
                ->url(guard_url('testing/testing/' . $testing->getRouteKey()))
                ->redirect();
        } catch (Exception $e) {
            return $this->response->message($e->getMessage())
                ->code(400)
                ->status('error')
                ->url(guard_url('/testing/testing'))
                ->redirect();
        }

    }

    /**
     * Show testing for editing.
     *
     * @param Request $request
     * @param Model   $testing
     *
     * @return Response
     */
    public function edit(TestingRequest $request, Testing $testing)
    {
        return $this->response->title(trans('app.edit') . ' ' . trans('testing::testing.name'))
            ->view('testing::testing.edit', true)
            ->data(compact('testing'))
            ->output();
    }

    /**
     * Update the testing.
     *
     * @param Request $request
     * @param Model   $testing
     *
     * @return Response
     */
    public function update(TestingRequest $request, Testing $testing)
    {
        try {
            $attributes = $request->all();

            $testing->update($attributes);
            return $this->response->message(trans('messages.success.updated', ['Module' => trans('testing::testing.name')]))
                ->code(204)
                ->status('success')
                ->url(guard_url('testing/testing/' . $testing->getRouteKey()))
                ->redirect();
        } catch (Exception $e) {
            return $this->response->message($e->getMessage())
                ->code(400)
                ->status('error')
                ->url(guard_url('testing/testing/' . $testing->getRouteKey()))
                ->redirect();
        }

    }

    /**
     * Remove the testing.
     *
     * @param Model   $testing
     *
     * @return Response
     */
    public function destroy(TestingRequest $request, Testing $testing)
    {
        try {

            $testing->delete();
            return $this->response->message(trans('messages.success.deleted', ['Module' => trans('testing::testing.name')]))
                ->code(202)
                ->status('success')
                ->url(guard_url('testing/testing/0'))
                ->redirect();

        } catch (Exception $e) {

            return $this->response->message($e->getMessage())
                ->code(400)
                ->status('error')
                ->url(guard_url('testing/testing/' . $testing->getRouteKey()))
                ->redirect();
        }

    }

    /**
     * Remove multiple testing.
     *
     * @param Model   $testing
     *
     * @return Response
     */
    public function delete(TestingRequest $request, $type)
    {
        try {
            $ids = hashids_decode($request->input('ids'));

            if ($type == 'purge') {
                $this->repository->purge($ids);
            } else {
                $this->repository->delete($ids);
            }

            return $this->response->message(trans('messages.success.deleted', ['Module' => trans('testing::testing.name')]))
                ->status("success")
                ->code(202)
                ->url(guard_url('testing/testing'))
                ->redirect();

        } catch (Exception $e) {

            return $this->response->message($e->getMessage())
                ->status("error")
                ->code(400)
                ->url(guard_url('/testing/testing'))
                ->redirect();
        }

    }

    /**
     * Restore deleted testings.
     *
     * @param Model   $testing
     *
     * @return Response
     */
    public function restore(TestingRequest $request)
    {
        try {
            $ids = hashids_decode($request->input('ids'));
            $this->repository->restore($ids);

            return $this->response->message(trans('messages.success.restore', ['Module' => trans('testing::testing.name')]))
                ->status("success")
                ->code(202)
                ->url(guard_url('/testing/testing'))
                ->redirect();

        } catch (Exception $e) {

            return $this->response->message($e->getMessage())
                ->status("error")
                ->code(400)
                ->url(guard_url('/testing/testing/'))
                ->redirect();
        }

    }

}
