<?php

namespace School\School\Http\Controllers;

use App\Http\Controllers\ResourceController as BaseController;
use Form;
use School\School\Http\Requests\SchoolRequest;
use School\School\Interfaces\SchoolRepositoryInterface;
use School\School\Models\School;

/**
 * Resource controller class for school.
 */
class SchoolResourceController extends BaseController
{

    /**
     * Initialize school resource controller.
     *
     * @param type SchoolRepositoryInterface $school
     *
     * @return null
     */
    public function __construct(SchoolRepositoryInterface $school)
    {
        parent::__construct();
        $this->repository = $school;
        $this->repository
            ->pushCriteria(\Litepie\Repository\Criteria\RequestCriteria::class)
            ->pushCriteria(\School\School\Repositories\Criteria\SchoolResourceCriteria::class);
    }

    /**
     * Display a list of school.
     *
     * @return Response
     */
    public function index(SchoolRequest $request)
    {
        $view = $this->response->theme->listView();

        if ($this->response->typeIs('json')) {
            $function = camel_case('get-' . $view);
            return $this->repository
                ->setPresenter(\School\School\Repositories\Presenter\SchoolPresenter::class)
                ->$function();
        }

        $schools = $this->repository->paginate();

        return $this->response->setMetaTitle(trans('school::school.names'))
            ->view('school::school.index', true)
            ->data(compact('schools', 'view'))
            ->output();
    }

    /**
     * Display school.
     *
     * @param Request $request
     * @param Model   $school
     *
     * @return Response
     */
    public function show(SchoolRequest $request, School $school)
    {

        if ($school->exists) {
            $view = 'school::school.show';
        } else {
            $view = 'school::school.new';
        }

        return $this->response->setMetaTitle(trans('app.view') . ' ' . trans('school::school.name'))
            ->data(compact('school'))
            ->view($view, true)
            ->output();
    }

    /**
     * Show the form for creating a new school.
     *
     * @param Request $request
     *
     * @return Response
     */
    public function create(SchoolRequest $request)
    {

        $school = $this->repository->newInstance([]);
        return $this->response->setMetaTitle(trans('app.new') . ' ' . trans('school::school.name')) 
            ->view('school::school.create', true) 
            ->data(compact('school'))
            ->output();
    }

    /**
     * Create new school.
     *
     * @param Request $request
     *
     * @return Response
     */
    public function store(SchoolRequest $request)
    {
        try {
            $attributes              = $request->all();
            $attributes['user_id']   = user_id();
            $attributes['user_type'] = user_type();
            $school                 = $this->repository->create($attributes);

            return $this->response->message(trans('messages.success.created', ['Module' => trans('school::school.name')]))
                ->code(204)
                ->status('success')
                ->url(guard_url('school/school/' . $school->getRouteKey()))
                ->redirect();
        } catch (Exception $e) {
            return $this->response->message($e->getMessage())
                ->code(400)
                ->status('error')
                ->url(guard_url('/school/school'))
                ->redirect();
        }

    }

    /**
     * Show school for editing.
     *
     * @param Request $request
     * @param Model   $school
     *
     * @return Response
     */
    public function edit(SchoolRequest $request, School $school)
    {
        return $this->response->setMetaTitle(trans('app.edit') . ' ' . trans('school::school.name'))
            ->view('school::school.edit', true)
            ->data(compact('school'))
            ->output();
    }

    /**
     * Update the school.
     *
     * @param Request $request
     * @param Model   $school
     *
     * @return Response
     */
    public function update(SchoolRequest $request, School $school)
    {
        try {
            $attributes = $request->all();

            $school->update($attributes);
            return $this->response->message(trans('messages.success.updated', ['Module' => trans('school::school.name')]))
                ->code(204)
                ->status('success')
                ->url(guard_url('school/school/' . $school->getRouteKey()))
                ->redirect();
        } catch (Exception $e) {
            return $this->response->message($e->getMessage())
                ->code(400)
                ->status('error')
                ->url(guard_url('school/school/' . $school->getRouteKey()))
                ->redirect();
        }

    }

    /**
     * Remove the school.
     *
     * @param Model   $school
     *
     * @return Response
     */
    public function destroy(SchoolRequest $request, School $school)
    {
        try {

            $school->delete();
            return $this->response->message(trans('messages.success.deleted', ['Module' => trans('school::school.name')]))
                ->code(202)
                ->status('success')
                ->url(guard_url('school/school/0'))
                ->redirect();

        } catch (Exception $e) {

            return $this->response->message($e->getMessage())
                ->code(400)
                ->status('error')
                ->url(guard_url('school/school/' . $school->getRouteKey()))
                ->redirect();
        }

    }

    /**
     * Remove multiple school.
     *
     * @param Model   $school
     *
     * @return Response
     */
    public function delete(SchoolRequest $request, $type)
    {
        try {
            $ids = hashids_decode($request->input('ids'));

            if ($type == 'purge') {
                $this->repository->purge($ids);
            } else {
                $this->repository->delete($ids);
            }

            return $this->response->message(trans('messages.success.deleted', ['Module' => trans('school::school.name')]))
                ->status("success")
                ->code(202)
                ->url(guard_url('school/school'))
                ->redirect();

        } catch (Exception $e) {

            return $this->response->message($e->getMessage())
                ->status("error")
                ->code(400)
                ->url(guard_url('/school/school'))
                ->redirect();
        }

    }

    /**
     * Restore deleted schools.
     *
     * @param Model   $school
     *
     * @return Response
     */
    public function restore(SchoolRequest $request)
    {
        try {
            $ids = hashids_decode($request->input('ids'));
            $this->repository->restore($ids);

            return $this->response->message(trans('messages.success.restore', ['Module' => trans('school::school.name')]))
                ->status("success")
                ->code(202)
                ->url(guard_url('/school/school'))
                ->redirect();

        } catch (Exception $e) {

            return $this->response->message($e->getMessage())
                ->status("error")
                ->code(400)
                ->url(guard_url('/school/school/'))
                ->redirect();
        }

    }

}
