<?php

namespace School\School\Http\Controllers;

use App\Http\Controllers\ResourceController as BaseController;
use Form;
use School\School\Http\Requests\StudentRequest;
use School\School\Interfaces\StudentRepositoryInterface;
use School\School\Models\Student;

/**
 * Resource controller class for student.
 */
class StudentResourceController extends BaseController
{

    /**
     * Initialize student resource controller.
     *
     * @param type StudentRepositoryInterface $student
     *
     * @return null
     */
    public function __construct(StudentRepositoryInterface $student)
    {
        parent::__construct();
        $this->repository = $student;
        $this->repository
            ->pushCriteria(\Litepie\Repository\Criteria\RequestCriteria::class)
            ->pushCriteria(\School\School\Repositories\Criteria\StudentResourceCriteria::class);
    }

    /**
     * Display a list of student.
     *
     * @return Response
     */
    public function index(StudentRequest $request)
    {
        $view = $this->response->theme->listView();

        if ($this->response->typeIs('json')) {
            $function = camel_case('get-' . $view);
            return $this->repository
                ->setPresenter(\School\School\Repositories\Presenter\StudentPresenter::class)
                ->$function();
        }

        $students = $this->repository->paginate();

        return $this->response->setMetaTitle(trans('school::student.names'))
            ->view('school::student.index', true)
            ->data(compact('students', 'view'))
            ->output();
    }

    /**
     * Display student.
     *
     * @param Request $request
     * @param Model   $student
     *
     * @return Response
     */
    public function show(StudentRequest $request, Student $student)
    {

        if ($student->exists) {
            $view = 'school::student.show';
        } else {
            $view = 'school::student.new';
        }

        return $this->response->setMetaTitle(trans('app.view') . ' ' . trans('school::student.name'))
            ->data(compact('student'))
            ->view($view, true)
            ->output();
    }

    /**
     * Show the form for creating a new student.
     *
     * @param Request $request
     *
     * @return Response
     */
    public function create(StudentRequest $request)
    {

        $student = $this->repository->newInstance([]);
        return $this->response->setMetaTitle(trans('app.new') . ' ' . trans('school::student.name')) 
            ->view('school::student.create', true) 
            ->data(compact('student'))
            ->output();
    }

    /**
     * Create new student.
     *
     * @param Request $request
     *
     * @return Response
     */
    public function store(StudentRequest $request)
    {
        try {
            $attributes              = $request->all();
            $attributes['user_id']   = user_id();
            $attributes['user_type'] = user_type();
            $student                 = $this->repository->create($attributes);

            return $this->response->message(trans('messages.success.created', ['Module' => trans('school::student.name')]))
                ->code(204)
                ->status('success')
                ->url(guard_url('school/student/' . $student->getRouteKey()))
                ->redirect();
        } catch (Exception $e) {
            return $this->response->message($e->getMessage())
                ->code(400)
                ->status('error')
                ->url(guard_url('/school/student'))
                ->redirect();
        }

    }

    /**
     * Show student for editing.
     *
     * @param Request $request
     * @param Model   $student
     *
     * @return Response
     */
    public function edit(StudentRequest $request, Student $student)
    {
        return $this->response->setMetaTitle(trans('app.edit') . ' ' . trans('school::student.name'))
            ->view('school::student.edit', true)
            ->data(compact('student'))
            ->output();
    }

    /**
     * Update the student.
     *
     * @param Request $request
     * @param Model   $student
     *
     * @return Response
     */
    public function update(StudentRequest $request, Student $student)
    {
        try {
            $attributes = $request->all();

            $student->update($attributes);
            return $this->response->message(trans('messages.success.updated', ['Module' => trans('school::student.name')]))
                ->code(204)
                ->status('success')
                ->url(guard_url('school/student/' . $student->getRouteKey()))
                ->redirect();
        } catch (Exception $e) {
            return $this->response->message($e->getMessage())
                ->code(400)
                ->status('error')
                ->url(guard_url('school/student/' . $student->getRouteKey()))
                ->redirect();
        }

    }

    /**
     * Remove the student.
     *
     * @param Model   $student
     *
     * @return Response
     */
    public function destroy(StudentRequest $request, Student $student)
    {
        try {

            $student->delete();
            return $this->response->message(trans('messages.success.deleted', ['Module' => trans('school::student.name')]))
                ->code(202)
                ->status('success')
                ->url(guard_url('school/student/0'))
                ->redirect();

        } catch (Exception $e) {

            return $this->response->message($e->getMessage())
                ->code(400)
                ->status('error')
                ->url(guard_url('school/student/' . $student->getRouteKey()))
                ->redirect();
        }

    }

    /**
     * Remove multiple student.
     *
     * @param Model   $student
     *
     * @return Response
     */
    public function delete(StudentRequest $request, $type)
    {
        try {
            $ids = hashids_decode($request->input('ids'));

            if ($type == 'purge') {
                $this->repository->purge($ids);
            } else {
                $this->repository->delete($ids);
            }

            return $this->response->message(trans('messages.success.deleted', ['Module' => trans('school::student.name')]))
                ->status("success")
                ->code(202)
                ->url(guard_url('school/student'))
                ->redirect();

        } catch (Exception $e) {

            return $this->response->message($e->getMessage())
                ->status("error")
                ->code(400)
                ->url(guard_url('/school/student'))
                ->redirect();
        }

    }

    /**
     * Restore deleted students.
     *
     * @param Model   $student
     *
     * @return Response
     */
    public function restore(StudentRequest $request)
    {
        try {
            $ids = hashids_decode($request->input('ids'));
            $this->repository->restore($ids);

            return $this->response->message(trans('messages.success.restore', ['Module' => trans('school::student.name')]))
                ->status("success")
                ->code(202)
                ->url(guard_url('/school/student'))
                ->redirect();

        } catch (Exception $e) {

            return $this->response->message($e->getMessage())
                ->status("error")
                ->code(400)
                ->url(guard_url('/school/student/'))
                ->redirect();
        }

    }

}
