<?php

namespace Testing\Testing\Http\Controllers;

use App\Http\Controllers\ResourceController as BaseController;
use Form;
use Testing\Testing\Http\Requests\NiceRequest;
use Testing\Testing\Interfaces\NiceRepositoryInterface;
use Testing\Testing\Models\Nice;

/**
 * Resource controller class for nice.
 */
class NiceResourceController extends BaseController
{

    /**
     * Initialize nice resource controller.
     *
     * @param type NiceRepositoryInterface $nice
     *
     * @return null
     */
    public function __construct(NiceRepositoryInterface $nice)
    {
        parent::__construct();
        $this->repository = $nice;
        $this->repository
            ->pushCriteria(\Litepie\Repository\Criteria\RequestCriteria::class)
            ->pushCriteria(\Testing\Testing\Repositories\Criteria\NiceResourceCriteria::class);
    }

    /**
     * Display a list of nice.
     *
     * @return Response
     */
    public function index(NiceRequest $request)
    {
        $view = $this->response->theme->listView();

        if ($this->response->typeIs('json')) {
            $function = camel_case('get-' . $view);
            return $this->repository
                ->setPresenter(\Testing\Testing\Repositories\Presenter\NicePresenter::class)
                ->$function();
        }

        $nices = $this->repository->paginate();

        return $this->response->title(trans('testing::nice.names'))
            ->view('testing::nice.index', true)
            ->data(compact('nices'))
            ->output();
    }

    /**
     * Display nice.
     *
     * @param Request $request
     * @param Model   $nice
     *
     * @return Response
     */
    public function show(NiceRequest $request, Nice $nice)
    {

        if ($nice->exists) {
            $view = 'testing::nice.show';
        } else {
            $view = 'testing::nice.new';
        }

        return $this->response->title(trans('app.view') . ' ' . trans('testing::nice.name'))
            ->data(compact('nice'))
            ->view($view, true)
            ->output();
    }

    /**
     * Show the form for creating a new nice.
     *
     * @param Request $request
     *
     * @return Response
     */
    public function create(NiceRequest $request)
    {

        $nice = $this->repository->newInstance([]);
        return $this->response->title(trans('app.new') . ' ' . trans('testing::nice.name')) 
            ->view('testing::nice.create', true) 
            ->data(compact('nice'))
            ->output();
    }

    /**
     * Create new nice.
     *
     * @param Request $request
     *
     * @return Response
     */
    public function store(NiceRequest $request)
    {
        try {
            $attributes              = $request->all();
            $attributes['user_id']   = user_id();
            $attributes['user_type'] = user_type();
            $nice                 = $this->repository->create($attributes);

            return $this->response->message(trans('messages.success.created', ['Module' => trans('testing::nice.name')]))
                ->code(204)
                ->status('success')
                ->url(guard_url('testing/nice/' . $nice->getRouteKey()))
                ->redirect();
        } catch (Exception $e) {
            return $this->response->message($e->getMessage())
                ->code(400)
                ->status('error')
                ->url(guard_url('/testing/nice'))
                ->redirect();
        }

    }

    /**
     * Show nice for editing.
     *
     * @param Request $request
     * @param Model   $nice
     *
     * @return Response
     */
    public function edit(NiceRequest $request, Nice $nice)
    {
        return $this->response->title(trans('app.edit') . ' ' . trans('testing::nice.name'))
            ->view('testing::nice.edit', true)
            ->data(compact('nice'))
            ->output();
    }

    /**
     * Update the nice.
     *
     * @param Request $request
     * @param Model   $nice
     *
     * @return Response
     */
    public function update(NiceRequest $request, Nice $nice)
    {
        try {
            $attributes = $request->all();

            $nice->update($attributes);
            return $this->response->message(trans('messages.success.updated', ['Module' => trans('testing::nice.name')]))
                ->code(204)
                ->status('success')
                ->url(guard_url('testing/nice/' . $nice->getRouteKey()))
                ->redirect();
        } catch (Exception $e) {
            return $this->response->message($e->getMessage())
                ->code(400)
                ->status('error')
                ->url(guard_url('testing/nice/' . $nice->getRouteKey()))
                ->redirect();
        }

    }

    /**
     * Remove the nice.
     *
     * @param Model   $nice
     *
     * @return Response
     */
    public function destroy(NiceRequest $request, Nice $nice)
    {
        try {

            $nice->delete();
            return $this->response->message(trans('messages.success.deleted', ['Module' => trans('testing::nice.name')]))
                ->code(202)
                ->status('success')
                ->url(guard_url('testing/nice/0'))
                ->redirect();

        } catch (Exception $e) {

            return $this->response->message($e->getMessage())
                ->code(400)
                ->status('error')
                ->url(guard_url('testing/nice/' . $nice->getRouteKey()))
                ->redirect();
        }

    }

    /**
     * Remove multiple nice.
     *
     * @param Model   $nice
     *
     * @return Response
     */
    public function delete(NiceRequest $request, $type)
    {
        try {
            $ids = hashids_decode($request->input('ids'));

            if ($type == 'purge') {
                $this->repository->purge($ids);
            } else {
                $this->repository->delete($ids);
            }

            return $this->response->message(trans('messages.success.deleted', ['Module' => trans('testing::nice.name')]))
                ->status("success")
                ->code(202)
                ->url(guard_url('testing/nice'))
                ->redirect();

        } catch (Exception $e) {

            return $this->response->message($e->getMessage())
                ->status("error")
                ->code(400)
                ->url(guard_url('/testing/nice'))
                ->redirect();
        }

    }

    /**
     * Restore deleted nices.
     *
     * @param Model   $nice
     *
     * @return Response
     */
    public function restore(NiceRequest $request)
    {
        try {
            $ids = hashids_decode($request->input('ids'));
            $this->repository->restore($ids);

            return $this->response->message(trans('messages.success.restore', ['Module' => trans('testing::nice.name')]))
                ->status("success")
                ->code(202)
                ->url(guard_url('/testing/nice'))
                ->redirect();

        } catch (Exception $e) {

            return $this->response->message($e->getMessage())
                ->status("error")
                ->code(400)
                ->url(guard_url('/testing/nice/'))
                ->redirect();
        }

    }

}
