<?php

namespace Litecms\BusListing\Http\Controllers;

use App\Http\Controllers\ResourceController as BaseController;
use Form;
use Litecms\BusListing\Http\Requests\BusListingRequest;
use Litecms\BusListing\Interfaces\BusListingRepositoryInterface;
use Litecms\BusListing\Models\BusListing;

/**
 * Resource controller class for bus_listing.
 */
class BusListingResourceController extends BaseController
{

    /**
     * Initialize bus_listing resource controller.
     *
     * @param type BusListingRepositoryInterface $bus_listing
     *
     * @return null
     */
    public function __construct(BusListingRepositoryInterface $bus_listing)
    {
        parent::__construct();
        $this->repository = $bus_listing;
        $this->repository
            ->pushCriteria(\Litepie\Repository\Criteria\RequestCriteria::class)
            ->pushCriteria(\Litecms\BusListing\Repositories\Criteria\BusListingResourceCriteria::class);
    }

    /**
     * Display a list of bus_listing.
     *
     * @return Response
     */
    public function index(BusListingRequest $request)
    {
        $view = $this->response->theme->listView();

        if ($this->response->typeIs('json')) {
            $function = camel_case('get-' . $view);
            return $this->repository
                ->setPresenter(\Litecms\BusListing\Repositories\Presenter\BusListingPresenter::class)
                ->$function();
        }

        $bus_listings = $this->repository->paginate();

        return $this->response->title(trans('bus_listing::bus_listing.names'))
            ->view('bus_listing::bus_listing.index', true)
            ->data(compact('bus_listings'))
            ->output();
    }

    /**
     * Display bus_listing.
     *
     * @param Request $request
     * @param Model   $bus_listing
     *
     * @return Response
     */
    public function show(BusListingRequest $request, BusListing $bus_listing)
    {

        if ($bus_listing->exists) {
            $view = 'bus_listing::bus_listing.show';
        } else {
            $view = 'bus_listing::bus_listing.new';
        }

        return $this->response->title(trans('app.view') . ' ' . trans('bus_listing::bus_listing.name'))
            ->data(compact('bus_listing'))
            ->view($view, true)
            ->output();
    }

    /**
     * Show the form for creating a new bus_listing.
     *
     * @param Request $request
     *
     * @return Response
     */
    public function create(BusListingRequest $request)
    {

        $bus_listing = $this->repository->newInstance([]);
        return $this->response->title(trans('app.new') . ' ' . trans('bus_listing::bus_listing.name')) 
            ->view('bus_listing::bus_listing.create', true) 
            ->data(compact('bus_listing'))
            ->output();
    }

    /**
     * Create new bus_listing.
     *
     * @param Request $request
     *
     * @return Response
     */
    public function store(BusListingRequest $request)
    {
        try {
            $attributes              = $request->all();
            $attributes['user_id']   = user_id();
            $attributes['user_type'] = user_type();
            $bus_listing                 = $this->repository->create($attributes);

            return $this->response->message(trans('messages.success.created', ['Module' => trans('bus_listing::bus_listing.name')]))
                ->code(204)
                ->status('success')
                ->url(guard_url('bus_listing/bus_listing/' . $bus_listing->getRouteKey()))
                ->redirect();
        } catch (Exception $e) {
            return $this->response->message($e->getMessage())
                ->code(400)
                ->status('error')
                ->url(guard_url('/bus_listing/bus_listing'))
                ->redirect();
        }

    }

    /**
     * Show bus_listing for editing.
     *
     * @param Request $request
     * @param Model   $bus_listing
     *
     * @return Response
     */
    public function edit(BusListingRequest $request, BusListing $bus_listing)
    {
        return $this->response->title(trans('app.edit') . ' ' . trans('bus_listing::bus_listing.name'))
            ->view('bus_listing::bus_listing.edit', true)
            ->data(compact('bus_listing'))
            ->output();
    }

    /**
     * Update the bus_listing.
     *
     * @param Request $request
     * @param Model   $bus_listing
     *
     * @return Response
     */
    public function update(BusListingRequest $request, BusListing $bus_listing)
    {
        try {
            $attributes = $request->all();

            $bus_listing->update($attributes);
            return $this->response->message(trans('messages.success.updated', ['Module' => trans('bus_listing::bus_listing.name')]))
                ->code(204)
                ->status('success')
                ->url(guard_url('bus_listing/bus_listing/' . $bus_listing->getRouteKey()))
                ->redirect();
        } catch (Exception $e) {
            return $this->response->message($e->getMessage())
                ->code(400)
                ->status('error')
                ->url(guard_url('bus_listing/bus_listing/' . $bus_listing->getRouteKey()))
                ->redirect();
        }

    }

    /**
     * Remove the bus_listing.
     *
     * @param Model   $bus_listing
     *
     * @return Response
     */
    public function destroy(BusListingRequest $request, BusListing $bus_listing)
    {
        try {

            $bus_listing->delete();
            return $this->response->message(trans('messages.success.deleted', ['Module' => trans('bus_listing::bus_listing.name')]))
                ->code(202)
                ->status('success')
                ->url(guard_url('bus_listing/bus_listing/0'))
                ->redirect();

        } catch (Exception $e) {

            return $this->response->message($e->getMessage())
                ->code(400)
                ->status('error')
                ->url(guard_url('bus_listing/bus_listing/' . $bus_listing->getRouteKey()))
                ->redirect();
        }

    }

    /**
     * Remove multiple bus_listing.
     *
     * @param Model   $bus_listing
     *
     * @return Response
     */
    public function delete(BusListingRequest $request, $type)
    {
        try {
            $ids = hashids_decode($request->input('ids'));

            if ($type == 'purge') {
                $this->repository->purge($ids);
            } else {
                $this->repository->delete($ids);
            }

            return $this->response->message(trans('messages.success.deleted', ['Module' => trans('bus_listing::bus_listing.name')]))
                ->status("success")
                ->code(202)
                ->url(guard_url('bus_listing/bus_listing'))
                ->redirect();

        } catch (Exception $e) {

            return $this->response->message($e->getMessage())
                ->status("error")
                ->code(400)
                ->url(guard_url('/bus_listing/bus_listing'))
                ->redirect();
        }

    }

    /**
     * Restore deleted bus_listings.
     *
     * @param Model   $bus_listing
     *
     * @return Response
     */
    public function restore(BusListingRequest $request)
    {
        try {
            $ids = hashids_decode($request->input('ids'));
            $this->repository->restore($ids);

            return $this->response->message(trans('messages.success.restore', ['Module' => trans('bus_listing::bus_listing.name')]))
                ->status("success")
                ->code(202)
                ->url(guard_url('/bus_listing/bus_listing'))
                ->redirect();

        } catch (Exception $e) {

            return $this->response->message($e->getMessage())
                ->status("error")
                ->code(400)
                ->url(guard_url('/bus_listing/bus_listing/'))
                ->redirect();
        }

    }

}
