<?php

namespace Litecms\BuyAd\Http\Controllers;

use App\Http\Controllers\ResourceController as BaseController;
use Form;
use Litecms\BuyAd\Http\Requests\BuyadsRequest;
use Litecms\BuyAd\Interfaces\BuyadsRepositoryInterface;
use Litecms\BuyAd\Models\Buyads;

/**
 * Resource controller class for buyads.
 */
class BuyadsResourceController extends BaseController
{

    /**
     * Initialize buyads resource controller.
     *
     * @param type BuyadsRepositoryInterface $buyads
     *
     * @return null
     */
    public function __construct(BuyadsRepositoryInterface $buyads)
    {
        parent::__construct();
        $this->repository = $buyads;
        $this->repository
            ->pushCriteria(\Litepie\Repository\Criteria\RequestCriteria::class)
            ->pushCriteria(\Litecms\BuyAd\Repositories\Criteria\BuyadsResourceCriteria::class);
    }

    /**
     * Display a list of buyads.
     *
     * @return Response
     */
    public function index(BuyadsRequest $request)
    {
        $view = $this->response->theme->listView();

        if ($this->response->typeIs('json')) {
            $function = camel_case('get-' . $view);
            return $this->repository
                ->setPresenter(\Litecms\BuyAd\Repositories\Presenter\BuyadsPresenter::class)
                ->$function();
        }

        $buyads = $this->repository->paginate();

        return $this->response->title(trans('buy_ad::buyads.names'))
            ->view('buy_ad::buyads.index', true)
            ->data(compact('buyads'))
            ->output();
    }

    /**
     * Display buyads.
     *
     * @param Request $request
     * @param Model   $buyads
     *
     * @return Response
     */
    public function show(BuyadsRequest $request, Buyads $buyads)
    {

        if ($buyads->exists) {
            $view = 'buy_ad::buyads.show';
        } else {
            $view = 'buy_ad::buyads.new';
        }

        return $this->response->title(trans('app.view') . ' ' . trans('buy_ad::buyads.name'))
            ->data(compact('buyads'))
            ->view($view, true)
            ->output();
    }

    /**
     * Show the form for creating a new buyads.
     *
     * @param Request $request
     *
     * @return Response
     */
    public function create(BuyadsRequest $request)
    {

        $buyads = $this->repository->newInstance([]);
        return $this->response->title(trans('app.new') . ' ' . trans('buy_ad::buyads.name')) 
            ->view('buy_ad::buyads.create', true) 
            ->data(compact('buyads'))
            ->output();
    }

    /**
     * Create new buyads.
     *
     * @param Request $request
     *
     * @return Response
     */
    public function store(BuyadsRequest $request)
    {
        try {
            $attributes              = $request->all();
            $attributes['user_id']   = user_id();
            $attributes['user_type'] = user_type();
            $buyads                 = $this->repository->create($attributes);

            return $this->response->message(trans('messages.success.created', ['Module' => trans('buy_ad::buyads.name')]))
                ->code(204)
                ->status('success')
                ->url(guard_url('buy_ad/buyads/' . $buyads->getRouteKey()))
                ->redirect();
        } catch (Exception $e) {
            return $this->response->message($e->getMessage())
                ->code(400)
                ->status('error')
                ->url(guard_url('/buy_ad/buyads'))
                ->redirect();
        }

    }

    /**
     * Show buyads for editing.
     *
     * @param Request $request
     * @param Model   $buyads
     *
     * @return Response
     */
    public function edit(BuyadsRequest $request, Buyads $buyads)
    {
        return $this->response->title(trans('app.edit') . ' ' . trans('buy_ad::buyads.name'))
            ->view('buy_ad::buyads.edit', true)
            ->data(compact('buyads'))
            ->output();
    }

    /**
     * Update the buyads.
     *
     * @param Request $request
     * @param Model   $buyads
     *
     * @return Response
     */
    public function update(BuyadsRequest $request, Buyads $buyads)
    {
        try {
            $attributes = $request->all();

            $buyads->update($attributes);
            return $this->response->message(trans('messages.success.updated', ['Module' => trans('buy_ad::buyads.name')]))
                ->code(204)
                ->status('success')
                ->url(guard_url('buy_ad/buyads/' . $buyads->getRouteKey()))
                ->redirect();
        } catch (Exception $e) {
            return $this->response->message($e->getMessage())
                ->code(400)
                ->status('error')
                ->url(guard_url('buy_ad/buyads/' . $buyads->getRouteKey()))
                ->redirect();
        }

    }

    /**
     * Remove the buyads.
     *
     * @param Model   $buyads
     *
     * @return Response
     */
    public function destroy(BuyadsRequest $request, Buyads $buyads)
    {
        try {

            $buyads->delete();
            return $this->response->message(trans('messages.success.deleted', ['Module' => trans('buy_ad::buyads.name')]))
                ->code(202)
                ->status('success')
                ->url(guard_url('buy_ad/buyads/0'))
                ->redirect();

        } catch (Exception $e) {

            return $this->response->message($e->getMessage())
                ->code(400)
                ->status('error')
                ->url(guard_url('buy_ad/buyads/' . $buyads->getRouteKey()))
                ->redirect();
        }

    }

    /**
     * Remove multiple buyads.
     *
     * @param Model   $buyads
     *
     * @return Response
     */
    public function delete(BuyadsRequest $request, $type)
    {
        try {
            $ids = hashids_decode($request->input('ids'));

            if ($type == 'purge') {
                $this->repository->purge($ids);
            } else {
                $this->repository->delete($ids);
            }

            return $this->response->message(trans('messages.success.deleted', ['Module' => trans('buy_ad::buyads.name')]))
                ->status("success")
                ->code(202)
                ->url(guard_url('buy_ad/buyads'))
                ->redirect();

        } catch (Exception $e) {

            return $this->response->message($e->getMessage())
                ->status("error")
                ->code(400)
                ->url(guard_url('/buy_ad/buyads'))
                ->redirect();
        }

    }

    /**
     * Restore deleted buyads.
     *
     * @param Model   $buyads
     *
     * @return Response
     */
    public function restore(BuyadsRequest $request)
    {
        try {
            $ids = hashids_decode($request->input('ids'));
            $this->repository->restore($ids);

            return $this->response->message(trans('messages.success.restore', ['Module' => trans('buy_ad::buyads.name')]))
                ->status("success")
                ->code(202)
                ->url(guard_url('/buy_ad/buyads'))
                ->redirect();

        } catch (Exception $e) {

            return $this->response->message($e->getMessage())
                ->status("error")
                ->code(400)
                ->url(guard_url('/buy_ad/buyads/'))
                ->redirect();
        }

    }

}
