<?php

namespace Order\Eee\Http\Controllers;

use Exception;
use Litepie\Http\Controllers\ResourceController as BaseController;
use Litepie\Database\RequestScope;
use Order\Eee\Forms\Order as OrderForm;
use Order\Eee\Http\Requests\OrderRequest;
use Order\Eee\Http\Resources\OrderResource;
use Order\Eee\Http\Resources\OrdersCollection;
use Order\Eee\Models\Order;
use Order\Eee\Scopes\OrderResourceScope;


/**
 * Resource controller class for order.
 */
class OrderResourceController extends BaseController
{

    /**
     * Initialize order resource controller.
     *
     *
     * @return null
     */
    public function __construct()
    {
        parent::__construct();
        $this->form = OrderForm::grouped(false)
                        ->setAttributes()
                        ->toArray();
        $this->modules = $this->modules(config('order.eee.modules'), 'eee', guard_url('eee'));
    }

    /**
     * Display a list of order.
     *
     * @return Response
     */
    public function index(OrderRequest $request)
    {

        $pageLimit = $request->input('pageLimit', config('database.pagination.limit'));
        $page = Order::pushScope(new RequestScope())
            ->pushScope(new OrderResourceScope())
            ->simplePaginate($pageLimit);

        $data = new OrdersCollection($page);

        $form = $this->form;
        $modules = $this->modules;

        return $this->response->setMetaTitle(trans('eee::order.names'))
            ->view('eee::order.index')
            ->data(compact('data', 'modules', 'form'))
            ->output();

    }

    /**
     * Display order.
     *
     * @param Request $request
     * @param Model   $order
     *
     * @return Response
     */
    public function show(OrderRequest $request, Order $model)
    {
        $form = $this->form;
        $modules = $this->modules;
        $data = new OrderResource($model);

        return $this->response
            ->setMetaTitle(trans('app.view') . ' ' . trans('eee::order.name'))
            ->data(compact('data', 'form', 'modules'))
            ->view('eee::order.show')
            ->output();

    }

    /**
     * Show the form for creating a new order.
     *
     * @param Request $request
     *
     * @return Response
     */
    public function create(OrderRequest $request, Order $model)
    {
        $form = $this->form;
        $modules = $this->modules;
        $data = new OrderResource($model);
        return $this->response->setMetaTitle(trans('app.new') . ' ' . trans('eee::order.name'))
            ->view('eee::order.create')
            ->data(compact('data', 'form', 'modules'))
            ->output();

    }

    /**
     * Create new order.
     *
     * @param Request $request
     *
     * @return Response
     */
    public function store(OrderRequest $request, Order $model)
    {
        try {
            $attributes = $request->all();
            $attributes['user_id'] = user_id();
            $attributes['user_type'] = user_type();
            $model = $model->create($attributes);
            $data = new OrderResource($model);

            return $this->response->message(trans('messages.success.created', ['Module' => trans('eee::order.name')]))
                ->code(204)
                ->data(compact('data'))
                ->status('success')
                ->url(guard_url('eee/order/' . $model->getRouteKey()))
                ->redirect();
        } catch (Exception $e) {
            return $this->response->message($e->getMessage())
                ->code(400)
                ->status('error')
                ->url(guard_url('/eee/order'))
                ->redirect();
        }

    }

    /**
     * Show order for editing.
     *
     * @param Request $request
     * @param Model   $order
     *
     * @return Response
     */
    public function edit(OrderRequest $request, Order $model)
    {
        $form = $this->form;
        $modules = $this->modules;
        $data = new OrderResource($model);

        return $this->response->setMetaTitle(trans('app.edit') . ' ' . trans('eee::order.name'))
            ->view('eee::order.edit')
            ->data(compact('data', 'form', 'modules'))
            ->output();

    }

    /**
     * Update the order.
     *
     * @param Request $request
     * @param Model   $order
     *
     * @return Response
     */
    public function update(OrderRequest $request, Order $model)
    {
        try {
            $attributes = $request->all();
            $model = $model->update($attributes);
            $data = new OrderResource($model);

            return $this->response->message(trans('messages.success.updated', ['Module' => trans('eee::order.name')]))
                ->code(204)
                ->status('success')
                ->data(compact('data'))
                ->url(guard_url('eee/order/' . $model->getRouteKey()))
                ->redirect();
        } catch (Exception $e) {
            return $this->response->message($e->getMessage())
                ->code(400)
                ->status('error')
                ->url(guard_url('eee/order/' .  $model->getRouteKey()))
                ->redirect();
        }

    }

    /**
     * Remove the order.
     *
     * @param Model   $order
     *
     * @return Response
     */
    public function destroy(OrderRequest $request, Order $model)
    {
        try {
            $model->delete();
            $data = new OrderResource($model);

            return $this->response->message(trans('messages.success.deleted', ['Module' => trans('eee::order.name')]))
                ->code(202)
                ->status('success')
                ->data(compact('data'))
                ->url(guard_url('eee/order/0'))
                ->redirect();

        } catch (Exception $e) {

            return $this->response->message($e->getMessage())
                ->code(400)
                ->status('error')
                ->url(guard_url('eee/order/' .  $model->getRouteKey()))
                ->redirect();
        }

    }
}
