<?php

namespace Order\Eee\Http\Controllers;

use Exception;
use Litepie\Http\Controllers\ResourceController as BaseController;
use Litepie\Database\RequestScope;
use Order\Eee\Forms\Product as ProductForm;
use Order\Eee\Http\Requests\ProductRequest;
use Order\Eee\Http\Resources\ProductResource;
use Order\Eee\Http\Resources\ProductsCollection;
use Order\Eee\Models\Product;
use Order\Eee\Scopes\ProductResourceScope;


/**
 * Resource controller class for product.
 */
class ProductResourceController extends BaseController
{

    /**
     * Initialize product resource controller.
     *
     *
     * @return null
     */
    public function __construct()
    {
        parent::__construct();
        $this->form = ProductForm::grouped(false)
                        ->setAttributes()
                        ->toArray();
        $this->modules = $this->modules(config('order.eee.modules'), 'eee', guard_url('eee'));
    }

    /**
     * Display a list of product.
     *
     * @return Response
     */
    public function index(ProductRequest $request)
    {

        $pageLimit = $request->input('pageLimit', config('database.pagination.limit'));
        $page = Product::pushScope(new RequestScope())
            ->pushScope(new ProductResourceScope())
            ->simplePaginate($pageLimit);

        $data = new ProductsCollection($page);

        $form = $this->form;
        $modules = $this->modules;

        return $this->response->setMetaTitle(trans('eee::product.names'))
            ->view('eee::product.index')
            ->data(compact('data', 'modules', 'form'))
            ->output();

    }

    /**
     * Display product.
     *
     * @param Request $request
     * @param Model   $product
     *
     * @return Response
     */
    public function show(ProductRequest $request, Product $model)
    {
        $form = $this->form;
        $modules = $this->modules;
        $data = new ProductResource($model);

        return $this->response
            ->setMetaTitle(trans('app.view') . ' ' . trans('eee::product.name'))
            ->data(compact('data', 'form', 'modules'))
            ->view('eee::product.show')
            ->output();

    }

    /**
     * Show the form for creating a new product.
     *
     * @param Request $request
     *
     * @return Response
     */
    public function create(ProductRequest $request, Product $model)
    {
        $form = $this->form;
        $modules = $this->modules;
        $data = new ProductResource($model);
        return $this->response->setMetaTitle(trans('app.new') . ' ' . trans('eee::product.name'))
            ->view('eee::product.create')
            ->data(compact('data', 'form', 'modules'))
            ->output();

    }

    /**
     * Create new product.
     *
     * @param Request $request
     *
     * @return Response
     */
    public function store(ProductRequest $request, Product $model)
    {
        try {
            $attributes = $request->all();
            $attributes['user_id'] = user_id();
            $attributes['user_type'] = user_type();
            $model = $model->create($attributes);
            $data = new ProductResource($model);

            return $this->response->message(trans('messages.success.created', ['Module' => trans('eee::product.name')]))
                ->code(204)
                ->data(compact('data'))
                ->status('success')
                ->url(guard_url('eee/product/' . $model->getRouteKey()))
                ->redirect();
        } catch (Exception $e) {
            return $this->response->message($e->getMessage())
                ->code(400)
                ->status('error')
                ->url(guard_url('/eee/product'))
                ->redirect();
        }

    }

    /**
     * Show product for editing.
     *
     * @param Request $request
     * @param Model   $product
     *
     * @return Response
     */
    public function edit(ProductRequest $request, Product $model)
    {
        $form = $this->form;
        $modules = $this->modules;
        $data = new ProductResource($model);

        return $this->response->setMetaTitle(trans('app.edit') . ' ' . trans('eee::product.name'))
            ->view('eee::product.edit')
            ->data(compact('data', 'form', 'modules'))
            ->output();

    }

    /**
     * Update the product.
     *
     * @param Request $request
     * @param Model   $product
     *
     * @return Response
     */
    public function update(ProductRequest $request, Product $model)
    {
        try {
            $attributes = $request->all();
            $model = $model->update($attributes);
            $data = new ProductResource($model);

            return $this->response->message(trans('messages.success.updated', ['Module' => trans('eee::product.name')]))
                ->code(204)
                ->status('success')
                ->data(compact('data'))
                ->url(guard_url('eee/product/' . $model->getRouteKey()))
                ->redirect();
        } catch (Exception $e) {
            return $this->response->message($e->getMessage())
                ->code(400)
                ->status('error')
                ->url(guard_url('eee/product/' .  $model->getRouteKey()))
                ->redirect();
        }

    }

    /**
     * Remove the product.
     *
     * @param Model   $product
     *
     * @return Response
     */
    public function destroy(ProductRequest $request, Product $model)
    {
        try {
            $model->delete();
            $data = new ProductResource($model);

            return $this->response->message(trans('messages.success.deleted', ['Module' => trans('eee::product.name')]))
                ->code(202)
                ->status('success')
                ->data(compact('data'))
                ->url(guard_url('eee/product/0'))
                ->redirect();

        } catch (Exception $e) {

            return $this->response->message($e->getMessage())
                ->code(400)
                ->status('error')
                ->url(guard_url('eee/product/' .  $model->getRouteKey()))
                ->redirect();
        }

    }
}
