<?php

namespace Bixo\Payroll\Http\Controllers;

use Exception;
use Litepie\Http\Controllers\ResourceController as BaseController;
use Bixo\Payroll\Forms\Adjustment as AdjustmentForm;
use Bixo\Payroll\Http\Requests\AdjustmentResourceRequest;
use Bixo\Payroll\Http\Resources\AdjustmentResource;
use Bixo\Payroll\Http\Resources\AdjustmentsCollection;
use Bixo\Payroll\Models\Adjustment;
use Bixo\Payroll\Actions\AdjustmentAction;
use Bixo\Payroll\Actions\AdjustmentActions;


/**
 * Resource controller class for adjustment.
 */
class AdjustmentResourceController extends BaseController
{

    /**
     * Initialize adjustment resource controller.
     *
     *
     * @return null
     */
    public function __construct()
    {
        parent::__construct();
        $this->middleware(function ($request, $next) {
            $this->form = AdjustmentForm::grouped(false)
                            ->setAttributes()
                            ->toArray();
            $this->modules = $this->modules(config('bixo.payroll.modules'), 'payroll', guard_url('payroll'));
            return $next($request);
        });
    }

    /**
     * Display a list of adjustment.
     *
     * @return Response
     */
    public function index(AdjustmentResourceRequest $request)
    {
        $request = $request->all();
        $page = AdjustmentActions::run('paginate', $request);

        $data = new AdjustmentsCollection($page);

        $form = $this->form;
        $modules = $this->modules;

        return $this->response->setMetaTitle(trans('payroll::adjustment.names'))
            ->view('payroll::adjustment.index')
            ->data(compact('data', 'modules', 'form'))
            ->output();

    }

    /**
     * Display adjustment.
     *
     * @param Request $request
     * @param Model   $adjustment
     *
     * @return Response
     */
    public function show(AdjustmentResourceRequest $request, Adjustment $model)
    {
        $form = $this->form;
        $modules = $this->modules;
        $data = new AdjustmentResource($model);
        return $this->response
            ->setMetaTitle(trans('app.view') . ' ' . trans('payroll::adjustment.name'))
            ->data(compact('data', 'form', 'modules'))
            ->view('payroll::adjustment.show')
            ->output();
    }

    /**
     * Show the form for creating a new adjustment.
     *
     * @param Request $request
     *
     * @return Response
     */
    public function create(AdjustmentResourceRequest $request, Adjustment $model)
    {
        $form = $this->form;
        $modules = $this->modules;
        $data = new AdjustmentResource($model);
        return $this->response->setMetaTitle(trans('app.new') . ' ' . trans('payroll::adjustment.name'))
            ->view('payroll::adjustment.create')
            ->data(compact('data', 'form', 'modules'))
            ->output();

    }

    /**
     * Create new adjustment.
     *
     * @param Request $request
     *
     * @return Response
     */
    public function store(AdjustmentResourceRequest $request, Adjustment $model)
    {
        try {
            $request = $request->all();
            $model = AdjustmentAction::run('store', $model, $request);
            $data = new AdjustmentResource($model);
            return $this->response->message(trans('messages.success.created', ['Module' => trans('payroll::adjustment.name')]))
                ->code(204)
                ->data(compact('data'))
                ->status('success')
                ->url(guard_url('payroll/adjustment/' . $model->getRouteKey()))
                ->redirect();
        } catch (Exception $e) {
            return $this->response->message($e->getMessage())
                ->code(400)
                ->status('error')
                ->url(guard_url('/payroll/adjustment'))
                ->redirect();
        }

    }

    /**
     * Show adjustment for editing.
     *
     * @param Request $request
     * @param Model   $adjustment
     *
     * @return Response
     */
    public function edit(AdjustmentResourceRequest $request, Adjustment $model)
    {
        $form = $this->form;
        $modules = $this->modules;
        $data = new AdjustmentResource($model);
        // return view('payroll::adjustment.edit', compact('data', 'form', 'modules'));

        return $this->response->setMetaTitle(trans('app.edit') . ' ' . trans('payroll::adjustment.name'))
            ->view('payroll::adjustment.edit')
            ->data(compact('data', 'form', 'modules'))
            ->output();

    }

    /**
     * Update the adjustment.
     *
     * @param Request $request
     * @param Model   $adjustment
     *
     * @return Response
     */
    public function update(AdjustmentResourceRequest $request, Adjustment $model)
    {
        try {
            $request = $request->all();
            $model = AdjustmentAction::run('update', $model, $request);
            $data = new AdjustmentResource($model);

            return $this->response->message(trans('messages.success.updated', ['Module' => trans('payroll::adjustment.name')]))
                ->code(204)
                ->status('success')
                ->data(compact('data'))
                ->url(guard_url('payroll/adjustment/' . $model->getRouteKey()))
                ->redirect();
        } catch (Exception $e) {
            return $this->response->message($e->getMessage())
                ->code(400)
                ->status('error')
                ->url(guard_url('payroll/adjustment/' .  $model->getRouteKey()))
                ->redirect();
        }

    }

    /**
     * Remove the adjustment.
     *
     * @param Model   $adjustment
     *
     * @return Response
     */
    public function destroy(AdjustmentResourceRequest $request, Adjustment $model)
    {
        try {

            $request = $request->all();
            $model = AdjustmentAction::run('destroy', $model, $request);
            $data = new AdjustmentResource($model);

            return $this->response->message(trans('messages.success.deleted', ['Module' => trans('payroll::adjustment.name')]))
                ->code(202)
                ->status('success')
                ->data(compact('data'))
                ->url(guard_url('payroll/adjustment/0'))
                ->redirect();

        } catch (Exception $e) {

            return $this->response->message($e->getMessage())
                ->code(400)
                ->status('error')
                ->url(guard_url('payroll/adjustment/' .  $model->getRouteKey()))
                ->redirect();
        }

    }
}
