<?php

namespace Bixo\Payroll\Http\Controllers;

use Exception;
use Bixo\Payroll\Actions\AdjustmentWorkflow;
use Bixo\Payroll\Http\Requests\AdjustmentWorkflowRequest;
use Bixo\Payroll\Http\Resources\AdjustmentResource;
use Bixo\Payroll\Models\Adjustment;
use Litepie\Http\Controllers\ActionController as BaseController;

/**
 * Workflow controller class.
 *
 */
class AdjustmentWorkflowController extends BaseController
{
    /**
     * Action controller function for adjustment.
     *
     * @param Model $adjustment
     * @param action next action for the adjustment.
     *
     * @return Response
     */
    public function __invoke(AdjustmentWorkflowRequest $request, Adjustment $adjustment, $trasition)
    {
        try {
            $request = $request->all();
            $adjustment = AdjustmentWorkflow::run($trasition, $adjustment, $request);
            $data = new AdjustmentResource($adjustment);
            return $this->response->message(trans('messages.success.updated', ['Module' => trans('payroll::adjustment.name')]))
                ->code(204)
                ->data(compact('data'))
                ->status('success')
                ->url(guard_url('payroll/adjustment/' . $adjustment->getRouteKey()))
                ->redirect();
        } catch (Exception $e) {
            return $this->response->message($e->getMessage())
                ->code(400)
                ->status('error')
                ->url(guard_url('payroll/adjustment/' . $adjustment->getRouteKey()))
                ->redirect();
        }
    }


    /**
     * Get the adjustment and user for the workflow.
     *
     * @param PublicRequest $request
     * @param string $adjustment
     * @param string $user
     *
     * @return Response
     */
    public function get(PublicRequest $request, $adjustment, $user)
    {
        if (!$request->hasValidSignature()) {
            abort(403, 'This url is expired!');
        }

        $auth = Auth::guard()->getProvider()->getModel();
        $user = $auth::findBySignedId($user);
        Auth::guard()->login($user);
        $adjustment = Adjustment::findBySignedId($adjustment);
        $transitions = $this->transitions($adjustment, $user);
        $user = user();
        Auth::logout();
        return $this->response->setMetaTitle(trans('payroll::adjustment.name') . ' ' . trans('Approvals'))
            ->view('payroll::public.adjustment.workflow')
            ->layout('mobile')
            ->data(compact('adjustment', 'user', 'transitions'))
            ->output();
    }

    /**
     * Display the workflow form for the adjustment.
     *
     * @param PublicRequest $request
     * @param string $adjustment
     * @param string $user
     *
     * @return Response
     */
    public function post(PublicRequest $request, $adjustment, $user)
    {
        if (!$request->hasValidSignature()) {
            abort(403, 'This url is expired!');
        }
        $transition = $request->transition;
        $auth = Auth::guard()->getProvider()->getModel();
        $user = $auth::findBySignedId($user);
        Auth::guard()->login($user);
        $adjustment = Adjustment::findBySignedId($adjustment);
        $user = user();
        $adjustment = AdjustmentWorkflow::run($transition, $adjustment, $request->all());
        Auth::logout();

        return response()->json(
            [
                'status' => 'success',
                'url' => $request->url(),
            ]
        );
    }

    private function transitions($adjustment, $user)
    {
        $transitions = [];
        foreach ($adjustment->workflow()->transitions($adjustment) as $key => $value) {
            $name = $value->getName();
            $array['url'] = URL::temporarySignedRoute('litecms.adjustment.workflow', now()->addMinutes(3), [
                'guard' => 'admin',
                'transition' => $name,
                'adjustment' => $adjustment->getSignedId(now()->addMinutes(3)),
                'user' => $user->getSignedId(now()->addMinutes(3)),
                'trans' => 'en',
            ]
            );
            $array['name'] = $name;
            $array['key'] = $name;
            $array['form'] = $adjustment->workflow()->form($value);
            $array['label'] = trans('payroll::adjustment.workflow.' . $name);
            $transitions[] = $array;
        }
        return $transitions;
    }
}
