<?php

namespace Bixo\Payroll\Http\Controllers;

use Exception;
use Litepie\Http\Controllers\ResourceController as BaseController;
use Bixo\Payroll\Forms\Commission as CommissionForm;
use Bixo\Payroll\Http\Requests\CommissionResourceRequest;
use Bixo\Payroll\Http\Resources\CommissionResource;
use Bixo\Payroll\Http\Resources\CommissionsCollection;
use Bixo\Payroll\Models\Commission;
use Bixo\Payroll\Actions\CommissionAction;
use Bixo\Payroll\Actions\CommissionActions;


/**
 * Resource controller class for commission.
 */
class CommissionResourceController extends BaseController
{

    /**
     * Initialize commission resource controller.
     *
     *
     * @return null
     */
    public function __construct()
    {
        parent::__construct();
        $this->middleware(function ($request, $next) {
            $this->form = CommissionForm::grouped(false)
                            ->setAttributes()
                            ->toArray();
            $this->modules = $this->modules(config('bixo.payroll.modules'), 'payroll', guard_url('payroll'));
            return $next($request);
        });
    }

    /**
     * Display a list of commission.
     *
     * @return Response
     */
    public function index(CommissionResourceRequest $request)
    {
        $request = $request->all();
        $page = CommissionActions::run('paginate', $request);

        $data = new CommissionsCollection($page);

        $form = $this->form;
        $modules = $this->modules;

        return $this->response->setMetaTitle(trans('payroll::commission.names'))
            ->view('payroll::commission.index')
            ->data(compact('data', 'modules', 'form'))
            ->output();

    }

    /**
     * Display commission.
     *
     * @param Request $request
     * @param Model   $commission
     *
     * @return Response
     */
    public function show(CommissionResourceRequest $request, Commission $model)
    {
        $form = $this->form;
        $modules = $this->modules;
        $data = new CommissionResource($model);
        return $this->response
            ->setMetaTitle(trans('app.view') . ' ' . trans('payroll::commission.name'))
            ->data(compact('data', 'form', 'modules'))
            ->view('payroll::commission.show')
            ->output();
    }

    /**
     * Show the form for creating a new commission.
     *
     * @param Request $request
     *
     * @return Response
     */
    public function create(CommissionResourceRequest $request, Commission $model)
    {
        $form = $this->form;
        $modules = $this->modules;
        $data = new CommissionResource($model);
        return $this->response->setMetaTitle(trans('app.new') . ' ' . trans('payroll::commission.name'))
            ->view('payroll::commission.create')
            ->data(compact('data', 'form', 'modules'))
            ->output();

    }

    /**
     * Create new commission.
     *
     * @param Request $request
     *
     * @return Response
     */
    public function store(CommissionResourceRequest $request, Commission $model)
    {
        try {
            $request = $request->all();
            $model = CommissionAction::run('store', $model, $request);
            $data = new CommissionResource($model);
            return $this->response->message(trans('messages.success.created', ['Module' => trans('payroll::commission.name')]))
                ->code(204)
                ->data(compact('data'))
                ->status('success')
                ->url(guard_url('payroll/commission/' . $model->getRouteKey()))
                ->redirect();
        } catch (Exception $e) {
            return $this->response->message($e->getMessage())
                ->code(400)
                ->status('error')
                ->url(guard_url('/payroll/commission'))
                ->redirect();
        }

    }

    /**
     * Show commission for editing.
     *
     * @param Request $request
     * @param Model   $commission
     *
     * @return Response
     */
    public function edit(CommissionResourceRequest $request, Commission $model)
    {
        $form = $this->form;
        $modules = $this->modules;
        $data = new CommissionResource($model);
        // return view('payroll::commission.edit', compact('data', 'form', 'modules'));

        return $this->response->setMetaTitle(trans('app.edit') . ' ' . trans('payroll::commission.name'))
            ->view('payroll::commission.edit')
            ->data(compact('data', 'form', 'modules'))
            ->output();

    }

    /**
     * Update the commission.
     *
     * @param Request $request
     * @param Model   $commission
     *
     * @return Response
     */
    public function update(CommissionResourceRequest $request, Commission $model)
    {
        try {
            $request = $request->all();
            $model = CommissionAction::run('update', $model, $request);
            $data = new CommissionResource($model);

            return $this->response->message(trans('messages.success.updated', ['Module' => trans('payroll::commission.name')]))
                ->code(204)
                ->status('success')
                ->data(compact('data'))
                ->url(guard_url('payroll/commission/' . $model->getRouteKey()))
                ->redirect();
        } catch (Exception $e) {
            return $this->response->message($e->getMessage())
                ->code(400)
                ->status('error')
                ->url(guard_url('payroll/commission/' .  $model->getRouteKey()))
                ->redirect();
        }

    }

    /**
     * Remove the commission.
     *
     * @param Model   $commission
     *
     * @return Response
     */
    public function destroy(CommissionResourceRequest $request, Commission $model)
    {
        try {

            $request = $request->all();
            $model = CommissionAction::run('destroy', $model, $request);
            $data = new CommissionResource($model);

            return $this->response->message(trans('messages.success.deleted', ['Module' => trans('payroll::commission.name')]))
                ->code(202)
                ->status('success')
                ->data(compact('data'))
                ->url(guard_url('payroll/commission/0'))
                ->redirect();

        } catch (Exception $e) {

            return $this->response->message($e->getMessage())
                ->code(400)
                ->status('error')
                ->url(guard_url('payroll/commission/' .  $model->getRouteKey()))
                ->redirect();
        }

    }
}
