<?php

namespace Bixo\Payroll\Http\Controllers;

use Exception;
use Bixo\Payroll\Actions\CommissionWorkflow;
use Bixo\Payroll\Http\Requests\CommissionWorkflowRequest;
use Bixo\Payroll\Http\Resources\CommissionResource;
use Bixo\Payroll\Models\Commission;
use Litepie\Http\Controllers\ActionController as BaseController;

/**
 * Workflow controller class.
 *
 */
class CommissionWorkflowController extends BaseController
{
    /**
     * Action controller function for commission.
     *
     * @param Model $commission
     * @param action next action for the commission.
     *
     * @return Response
     */
    public function __invoke(CommissionWorkflowRequest $request, Commission $commission, $trasition)
    {
        try {
            $request = $request->all();
            $commission = CommissionWorkflow::run($trasition, $commission, $request);
            $data = new CommissionResource($commission);
            return $this->response->message(trans('messages.success.updated', ['Module' => trans('payroll::commission.name')]))
                ->code(204)
                ->data(compact('data'))
                ->status('success')
                ->url(guard_url('payroll/commission/' . $commission->getRouteKey()))
                ->redirect();
        } catch (Exception $e) {
            return $this->response->message($e->getMessage())
                ->code(400)
                ->status('error')
                ->url(guard_url('payroll/commission/' . $commission->getRouteKey()))
                ->redirect();
        }
    }


    /**
     * Get the commission and user for the workflow.
     *
     * @param PublicRequest $request
     * @param string $commission
     * @param string $user
     *
     * @return Response
     */
    public function get(PublicRequest $request, $commission, $user)
    {
        if (!$request->hasValidSignature()) {
            abort(403, 'This url is expired!');
        }

        $auth = Auth::guard()->getProvider()->getModel();
        $user = $auth::findBySignedId($user);
        Auth::guard()->login($user);
        $commission = Commission::findBySignedId($commission);
        $transitions = $this->transitions($commission, $user);
        $user = user();
        Auth::logout();
        return $this->response->setMetaTitle(trans('payroll::commission.name') . ' ' . trans('Approvals'))
            ->view('payroll::public.commission.workflow')
            ->layout('mobile')
            ->data(compact('commission', 'user', 'transitions'))
            ->output();
    }

    /**
     * Display the workflow form for the commission.
     *
     * @param PublicRequest $request
     * @param string $commission
     * @param string $user
     *
     * @return Response
     */
    public function post(PublicRequest $request, $commission, $user)
    {
        if (!$request->hasValidSignature()) {
            abort(403, 'This url is expired!');
        }
        $transition = $request->transition;
        $auth = Auth::guard()->getProvider()->getModel();
        $user = $auth::findBySignedId($user);
        Auth::guard()->login($user);
        $commission = Commission::findBySignedId($commission);
        $user = user();
        $commission = CommissionWorkflow::run($transition, $commission, $request->all());
        Auth::logout();

        return response()->json(
            [
                'status' => 'success',
                'url' => $request->url(),
            ]
        );
    }

    private function transitions($commission, $user)
    {
        $transitions = [];
        foreach ($commission->workflow()->transitions($commission) as $key => $value) {
            $name = $value->getName();
            $array['url'] = URL::temporarySignedRoute('litecms.commission.workflow', now()->addMinutes(3), [
                'guard' => 'admin',
                'transition' => $name,
                'commission' => $commission->getSignedId(now()->addMinutes(3)),
                'user' => $user->getSignedId(now()->addMinutes(3)),
                'trans' => 'en',
            ]
            );
            $array['name'] = $name;
            $array['key'] = $name;
            $array['form'] = $commission->workflow()->form($value);
            $array['label'] = trans('payroll::commission.workflow.' . $name);
            $transitions[] = $array;
        }
        return $transitions;
    }
}
