<?php

namespace Bixo\Payroll\Http\Controllers;

use Exception;
use Litepie\Http\Controllers\ResourceController as BaseController;
use Bixo\Payroll\Forms\Payroll as PayrollForm;
use Bixo\Payroll\Http\Requests\PayrollResourceRequest;
use Bixo\Payroll\Http\Resources\PayrollResource;
use Bixo\Payroll\Http\Resources\PayrollsCollection;
use Bixo\Payroll\Models\Payroll;
use Bixo\Payroll\Actions\PayrollAction;
use Bixo\Payroll\Actions\PayrollActions;


/**
 * Resource controller class for payroll.
 */
class PayrollResourceController extends BaseController
{

    /**
     * Initialize payroll resource controller.
     *
     *
     * @return null
     */
    public function __construct()
    {
        parent::__construct();
        $this->middleware(function ($request, $next) {
            $this->form = PayrollForm::grouped(false)
                            ->setAttributes()
                            ->toArray();
            $this->modules = $this->modules(config('bixo.payroll.modules'), 'payroll', guard_url('payroll'));
            return $next($request);
        });
    }

    /**
     * Display a list of payroll.
     *
     * @return Response
     */
    public function index(PayrollResourceRequest $request)
    {
        $request = $request->all();
        $page = PayrollActions::run('paginate', $request);

        $data = new PayrollsCollection($page);

        $form = $this->form;
        $modules = $this->modules;

        return $this->response->setMetaTitle(trans('payroll::payroll.names'))
            ->view('payroll::payroll.index')
            ->data(compact('data', 'modules', 'form'))
            ->output();

    }

    /**
     * Display payroll.
     *
     * @param Request $request
     * @param Model   $payroll
     *
     * @return Response
     */
    public function show(PayrollResourceRequest $request, Payroll $model)
    {
        $form = $this->form;
        $modules = $this->modules;
        $data = new PayrollResource($model);
        return $this->response
            ->setMetaTitle(trans('app.view') . ' ' . trans('payroll::payroll.name'))
            ->data(compact('data', 'form', 'modules'))
            ->view('payroll::payroll.show')
            ->output();
    }

    /**
     * Show the form for creating a new payroll.
     *
     * @param Request $request
     *
     * @return Response
     */
    public function create(PayrollResourceRequest $request, Payroll $model)
    {
        $form = $this->form;
        $modules = $this->modules;
        $data = new PayrollResource($model);
        return $this->response->setMetaTitle(trans('app.new') . ' ' . trans('payroll::payroll.name'))
            ->view('payroll::payroll.create')
            ->data(compact('data', 'form', 'modules'))
            ->output();

    }

    /**
     * Create new payroll.
     *
     * @param Request $request
     *
     * @return Response
     */
    public function store(PayrollResourceRequest $request, Payroll $model)
    {
        try {
            $request = $request->all();
            $model = PayrollAction::run('store', $model, $request);
            $data = new PayrollResource($model);
            return $this->response->message(trans('messages.success.created', ['Module' => trans('payroll::payroll.name')]))
                ->code(204)
                ->data(compact('data'))
                ->status('success')
                ->url(guard_url('payroll/payroll/' . $model->getRouteKey()))
                ->redirect();
        } catch (Exception $e) {
            return $this->response->message($e->getMessage())
                ->code(400)
                ->status('error')
                ->url(guard_url('/payroll/payroll'))
                ->redirect();
        }

    }

    /**
     * Show payroll for editing.
     *
     * @param Request $request
     * @param Model   $payroll
     *
     * @return Response
     */
    public function edit(PayrollResourceRequest $request, Payroll $model)
    {
        $form = $this->form;
        $modules = $this->modules;
        $data = new PayrollResource($model);
        // return view('payroll::payroll.edit', compact('data', 'form', 'modules'));

        return $this->response->setMetaTitle(trans('app.edit') . ' ' . trans('payroll::payroll.name'))
            ->view('payroll::payroll.edit')
            ->data(compact('data', 'form', 'modules'))
            ->output();

    }

    /**
     * Update the payroll.
     *
     * @param Request $request
     * @param Model   $payroll
     *
     * @return Response
     */
    public function update(PayrollResourceRequest $request, Payroll $model)
    {
        try {
            $request = $request->all();
            $model = PayrollAction::run('update', $model, $request);
            $data = new PayrollResource($model);

            return $this->response->message(trans('messages.success.updated', ['Module' => trans('payroll::payroll.name')]))
                ->code(204)
                ->status('success')
                ->data(compact('data'))
                ->url(guard_url('payroll/payroll/' . $model->getRouteKey()))
                ->redirect();
        } catch (Exception $e) {
            return $this->response->message($e->getMessage())
                ->code(400)
                ->status('error')
                ->url(guard_url('payroll/payroll/' .  $model->getRouteKey()))
                ->redirect();
        }

    }

    /**
     * Remove the payroll.
     *
     * @param Model   $payroll
     *
     * @return Response
     */
    public function destroy(PayrollResourceRequest $request, Payroll $model)
    {
        try {

            $request = $request->all();
            $model = PayrollAction::run('destroy', $model, $request);
            $data = new PayrollResource($model);

            return $this->response->message(trans('messages.success.deleted', ['Module' => trans('payroll::payroll.name')]))
                ->code(202)
                ->status('success')
                ->data(compact('data'))
                ->url(guard_url('payroll/payroll/0'))
                ->redirect();

        } catch (Exception $e) {

            return $this->response->message($e->getMessage())
                ->code(400)
                ->status('error')
                ->url(guard_url('payroll/payroll/' .  $model->getRouteKey()))
                ->redirect();
        }

    }
}
