<?php

namespace Bixo\Payroll\Http\Controllers;

use Exception;
use Bixo\Payroll\Actions\PayrollWorkflow;
use Bixo\Payroll\Http\Requests\PayrollWorkflowRequest;
use Bixo\Payroll\Http\Resources\PayrollResource;
use Bixo\Payroll\Models\Payroll;
use Litepie\Http\Controllers\ActionController as BaseController;

/**
 * Workflow controller class.
 *
 */
class PayrollWorkflowController extends BaseController
{
    /**
     * Action controller function for payroll.
     *
     * @param Model $payroll
     * @param action next action for the payroll.
     *
     * @return Response
     */
    public function __invoke(PayrollWorkflowRequest $request, Payroll $payroll, $trasition)
    {
        try {
            $request = $request->all();
            $payroll = PayrollWorkflow::run($trasition, $payroll, $request);
            $data = new PayrollResource($payroll);
            return $this->response->message(trans('messages.success.updated', ['Module' => trans('payroll::payroll.name')]))
                ->code(204)
                ->data(compact('data'))
                ->status('success')
                ->url(guard_url('payroll/payroll/' . $payroll->getRouteKey()))
                ->redirect();
        } catch (Exception $e) {
            return $this->response->message($e->getMessage())
                ->code(400)
                ->status('error')
                ->url(guard_url('payroll/payroll/' . $payroll->getRouteKey()))
                ->redirect();
        }
    }


    /**
     * Get the payroll and user for the workflow.
     *
     * @param PublicRequest $request
     * @param string $payroll
     * @param string $user
     *
     * @return Response
     */
    public function get(PublicRequest $request, $payroll, $user)
    {
        if (!$request->hasValidSignature()) {
            abort(403, 'This url is expired!');
        }

        $auth = Auth::guard()->getProvider()->getModel();
        $user = $auth::findBySignedId($user);
        Auth::guard()->login($user);
        $payroll = Payroll::findBySignedId($payroll);
        $transitions = $this->transitions($payroll, $user);
        $user = user();
        Auth::logout();
        return $this->response->setMetaTitle(trans('payroll::payroll.name') . ' ' . trans('Approvals'))
            ->view('payroll::public.payroll.workflow')
            ->layout('mobile')
            ->data(compact('payroll', 'user', 'transitions'))
            ->output();
    }

    /**
     * Display the workflow form for the payroll.
     *
     * @param PublicRequest $request
     * @param string $payroll
     * @param string $user
     *
     * @return Response
     */
    public function post(PublicRequest $request, $payroll, $user)
    {
        if (!$request->hasValidSignature()) {
            abort(403, 'This url is expired!');
        }
        $transition = $request->transition;
        $auth = Auth::guard()->getProvider()->getModel();
        $user = $auth::findBySignedId($user);
        Auth::guard()->login($user);
        $payroll = Payroll::findBySignedId($payroll);
        $user = user();
        $payroll = PayrollWorkflow::run($transition, $payroll, $request->all());
        Auth::logout();

        return response()->json(
            [
                'status' => 'success',
                'url' => $request->url(),
            ]
        );
    }

    private function transitions($payroll, $user)
    {
        $transitions = [];
        foreach ($payroll->workflow()->transitions($payroll) as $key => $value) {
            $name = $value->getName();
            $array['url'] = URL::temporarySignedRoute('litecms.payroll.workflow', now()->addMinutes(3), [
                'guard' => 'admin',
                'transition' => $name,
                'payroll' => $payroll->getSignedId(now()->addMinutes(3)),
                'user' => $user->getSignedId(now()->addMinutes(3)),
                'trans' => 'en',
            ]
            );
            $array['name'] = $name;
            $array['key'] = $name;
            $array['form'] = $payroll->workflow()->form($value);
            $array['label'] = trans('payroll::payroll.workflow.' . $name);
            $transitions[] = $array;
        }
        return $transitions;
    }
}
