<?php

namespace Bixo\Payroll\Http\Controllers;

use Exception;
use Litepie\Http\Controllers\ResourceController as BaseController;
use Bixo\Payroll\Forms\Salary as SalaryForm;
use Bixo\Payroll\Http\Requests\SalaryResourceRequest;
use Bixo\Payroll\Http\Resources\SalaryResource;
use Bixo\Payroll\Http\Resources\SalariesCollection;
use Bixo\Payroll\Models\Salary;
use Bixo\Payroll\Actions\SalaryAction;
use Bixo\Payroll\Actions\SalaryActions;


/**
 * Resource controller class for salary.
 */
class SalaryResourceController extends BaseController
{

    /**
     * Initialize salary resource controller.
     *
     *
     * @return null
     */
    public function __construct()
    {
        parent::__construct();
        $this->middleware(function ($request, $next) {
            $this->form = SalaryForm::grouped(false)
                            ->setAttributes()
                            ->toArray();
            $this->modules = $this->modules(config('bixo.payroll.modules'), 'payroll', guard_url('payroll'));
            return $next($request);
        });
    }

    /**
     * Display a list of salary.
     *
     * @return Response
     */
    public function index(SalaryResourceRequest $request)
    {
        $request = $request->all();
        $page = SalaryActions::run('paginate', $request);

        $data = new SalariesCollection($page);

        $form = $this->form;
        $modules = $this->modules;

        return $this->response->setMetaTitle(trans('payroll::salary.names'))
            ->view('payroll::salary.index')
            ->data(compact('data', 'modules', 'form'))
            ->output();

    }

    /**
     * Display salary.
     *
     * @param Request $request
     * @param Model   $salary
     *
     * @return Response
     */
    public function show(SalaryResourceRequest $request, Salary $model)
    {
        $form = $this->form;
        $modules = $this->modules;
        $data = new SalaryResource($model);
        return $this->response
            ->setMetaTitle(trans('app.view') . ' ' . trans('payroll::salary.name'))
            ->data(compact('data', 'form', 'modules'))
            ->view('payroll::salary.show')
            ->output();
    }

    /**
     * Show the form for creating a new salary.
     *
     * @param Request $request
     *
     * @return Response
     */
    public function create(SalaryResourceRequest $request, Salary $model)
    {
        $form = $this->form;
        $modules = $this->modules;
        $data = new SalaryResource($model);
        return $this->response->setMetaTitle(trans('app.new') . ' ' . trans('payroll::salary.name'))
            ->view('payroll::salary.create')
            ->data(compact('data', 'form', 'modules'))
            ->output();

    }

    /**
     * Create new salary.
     *
     * @param Request $request
     *
     * @return Response
     */
    public function store(SalaryResourceRequest $request, Salary $model)
    {
        try {
            $request = $request->all();
            $model = SalaryAction::run('store', $model, $request);
            $data = new SalaryResource($model);
            return $this->response->message(trans('messages.success.created', ['Module' => trans('payroll::salary.name')]))
                ->code(204)
                ->data(compact('data'))
                ->status('success')
                ->url(guard_url('payroll/salary/' . $model->getRouteKey()))
                ->redirect();
        } catch (Exception $e) {
            return $this->response->message($e->getMessage())
                ->code(400)
                ->status('error')
                ->url(guard_url('/payroll/salary'))
                ->redirect();
        }

    }

    /**
     * Show salary for editing.
     *
     * @param Request $request
     * @param Model   $salary
     *
     * @return Response
     */
    public function edit(SalaryResourceRequest $request, Salary $model)
    {
        $form = $this->form;
        $modules = $this->modules;
        $data = new SalaryResource($model);
        // return view('payroll::salary.edit', compact('data', 'form', 'modules'));

        return $this->response->setMetaTitle(trans('app.edit') . ' ' . trans('payroll::salary.name'))
            ->view('payroll::salary.edit')
            ->data(compact('data', 'form', 'modules'))
            ->output();

    }

    /**
     * Update the salary.
     *
     * @param Request $request
     * @param Model   $salary
     *
     * @return Response
     */
    public function update(SalaryResourceRequest $request, Salary $model)
    {
        try {
            $request = $request->all();
            $model = SalaryAction::run('update', $model, $request);
            $data = new SalaryResource($model);

            return $this->response->message(trans('messages.success.updated', ['Module' => trans('payroll::salary.name')]))
                ->code(204)
                ->status('success')
                ->data(compact('data'))
                ->url(guard_url('payroll/salary/' . $model->getRouteKey()))
                ->redirect();
        } catch (Exception $e) {
            return $this->response->message($e->getMessage())
                ->code(400)
                ->status('error')
                ->url(guard_url('payroll/salary/' .  $model->getRouteKey()))
                ->redirect();
        }

    }

    /**
     * Remove the salary.
     *
     * @param Model   $salary
     *
     * @return Response
     */
    public function destroy(SalaryResourceRequest $request, Salary $model)
    {
        try {

            $request = $request->all();
            $model = SalaryAction::run('destroy', $model, $request);
            $data = new SalaryResource($model);

            return $this->response->message(trans('messages.success.deleted', ['Module' => trans('payroll::salary.name')]))
                ->code(202)
                ->status('success')
                ->data(compact('data'))
                ->url(guard_url('payroll/salary/0'))
                ->redirect();

        } catch (Exception $e) {

            return $this->response->message($e->getMessage())
                ->code(400)
                ->status('error')
                ->url(guard_url('payroll/salary/' .  $model->getRouteKey()))
                ->redirect();
        }

    }
}
